
#!/bin/bash

# Script de inicialização do Gerenciador de Currículos

echo "🚀 Inicializando Gerenciador de Currículos..."

# Verificar se Docker está instalado
if ! command -v docker &> /dev/null; then
    echo "❌ Docker não encontrado. Por favor, instale o Docker primeiro."
    exit 1
fi

# Criar rede se não existir
docker network create curriculos_network 2>/dev/null || true

# Criar volumes se não existirem
docker volume create mysql_data 2>/dev/null || true
docker volume create app_sessions 2>/dev/null || true

# Parar containers existentes
echo "🛑 Parando containers existentes..."
docker stop gerenciador_curriculos_app 2>/dev/null || true
docker stop gerenciador_curriculos_mysql 2>/dev/null || true
docker stop gerenciador_curriculos_phpmyadmin 2>/dev/null || true

# Remover containers existentes
echo "🗑️ Removendo containers antigos..."
docker rm gerenciador_curriculos_app 2>/dev/null || true
docker rm gerenciador_curriculos_mysql 2>/dev/null || true
docker rm gerenciador_curriculos_phpmyadmin 2>/dev/null || true

# Iniciar MySQL
echo "🗄️ Iniciando MySQL..."
docker run -d \
  --name gerenciador_curriculos_mysql \
  --network curriculos_network \
  -e MYSQL_ROOT_PASSWORD=root_password \
  -e MYSQL_DATABASE=gerenciador_curriculos \
  -e MYSQL_USER=app_user \
  -e MYSQL_PASSWORD=app_password \
  -v mysql_data:/var/lib/mysql \
  -v "$(pwd)/scripts/init.sql:/docker-entrypoint-initdb.d/init.sql:ro" \
  -p 3307:3306 \
  mysql:8.0

# Aguardar MySQL inicializar
echo "⏳ Aguardando MySQL inicializar..."
sleep 30

# Build da aplicação
echo "🔨 Building aplicação..."
docker build -t gerenciador_curriculos:latest .

# Iniciar aplicação
echo "🌐 Iniciando aplicação..."
docker run -d \
  --name gerenciador_curriculos_app \
  --network curriculos_network \
  -e APP_NAME="Gerenciador de Currículos" \
  -e APP_URL=http://localhost:8080 \
  -e APP_DEBUG=true \
  -e DB_HOST=gerenciador_curriculos_mysql \
  -e DB_NAME=gerenciador_curriculos \
  -e DB_USER=root \
  -e DB_PASS=root_password \
  -v "$(pwd)/public/uploads:/var/www/html/public/uploads" \
  -v "$(pwd)/logs:/var/www/html/logs" \
  -v app_sessions:/tmp \
  -p 8080:80 \
  gerenciador_curriculos:latest

# Iniciar phpMyAdmin (opcional)
echo "📊 Iniciando phpMyAdmin..."
docker run -d \
  --name gerenciador_curriculos_phpmyadmin \
  --network curriculos_network \
  -e PMA_HOST=gerenciador_curriculos_mysql \
  -e PMA_PORT=3306 \
  -e PMA_USER=root \
  -e PMA_PASSWORD=root_password \
  -e MYSQL_ROOT_PASSWORD=root_password \
  -p 8081:80 \
  phpmyadmin:5.2-apache

echo ""
echo "✅ Sistema iniciado com sucesso!"
echo ""
echo "📱 Aplicação: http://localhost:8080"
echo "🗄️ phpMyAdmin: http://localhost:8081"
echo ""
echo "👤 Login padrão:"
echo "   Email: admin@empresa.com"
echo "   Senha: password"
echo ""
echo "📝 Logs:"
echo "   docker logs gerenciador_curriculos_app"
echo "   docker logs gerenciador_curriculos_mysql"
echo ""
echo "🛑 Para parar:"
echo "   ./stop.sh"
