
<?php

class Router {
    private $routes = [];
    private $notFoundHandler;
    private $middleware = [];
    
    public function __construct() {
        $this->notFoundHandler = function() {
            http_response_code(404);
            require VIEWS_PATH . '/errors/404.php';
        };
    }
    
    // Adicionar rota GET
    public function get($path, $handler) {
        $this->addRoute('GET', $path, $handler);
    }
    
    // Adicionar rota POST
    public function post($path, $handler) {
        $this->addRoute('POST', $path, $handler);
    }
    
    // Adicionar rota PUT
    public function put($path, $handler) {
        $this->addRoute('PUT', $path, $handler);
    }
    
    // Adicionar rota DELETE
    public function delete($path, $handler) {
        $this->addRoute('DELETE', $path, $handler);
    }
    
    // Adicionar middleware
    public function middleware($middleware) {
        $this->middleware[] = $middleware;
        return $this;
    }
    
    // Adicionar rota
    private function addRoute($method, $path, $handler) {
        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'handler' => $handler,
            'middleware' => $this->middleware
        ];
        $this->middleware = []; // Reset middleware após adicionar rota
    }
    
    // Executar roteamento
    public function run() {
        $requestMethod = $_SERVER['REQUEST_METHOD'];
        $requestUri = $_SERVER['REQUEST_URI'];
        
        // Remover query string
        $requestUri = strtok($requestUri, '?');
        
        // Remover trailing slash (exceto para root)
        if ($requestUri !== '/' && substr($requestUri, -1) === '/') {
            $requestUri = rtrim($requestUri, '/');
        }
        
        foreach ($this->routes as $route) {
            if ($route['method'] === $requestMethod && $this->matchPath($route['path'], $requestUri)) {
                // Executar middleware
                foreach ($route['middleware'] as $middleware) {
                    if (is_callable($middleware)) {
                        $result = call_user_func($middleware);
                        if ($result === false) {
                            return; // Middleware impediu execução
                        }
                    }
                }
                
                // Extrair parâmetros
                $params = $this->extractParams($route['path'], $requestUri);
                
                // Executar handler
                if (is_callable($route['handler'])) {
                    call_user_func($route['handler'], $params);
                } elseif (is_string($route['handler'])) {
                    $this->callControllerAction($route['handler'], $params);
                }
                
                return;
            }
        }
        
        // Rota não encontrada
        call_user_func($this->notFoundHandler);
    }
    
    // Verificar se o caminho corresponde
    private function matchPath($routePath, $requestUri) {
        $routePattern = preg_replace('/\{([^}]+)\}/', '([^/]+)', $routePath);
        $routePattern = '#^' . $routePattern . '$#';
        
        return preg_match($routePattern, $requestUri);
    }
    
    // Extrair parâmetros da URL
    private function extractParams($routePath, $requestUri) {
        $params = [];
        
        preg_match_all('/\{([^}]+)\}/', $routePath, $matches);
        $paramNames = $matches[1];
        
        $routePattern = preg_replace('/\{([^}]+)\}/', '([^/]+)', $routePath);
        $routePattern = '#^' . $routePattern . '$#';
        
        if (preg_match($routePattern, $requestUri, $values)) {
            array_shift($values); // Remove primeira correspondência (URL completa)
            
            for ($i = 0; $i < count($paramNames); $i++) {
                $params[$paramNames[$i]] = $values[$i] ?? null;
            }
        }
        
        return $params;
    }
    
    // Chamar ação do controller
    private function callControllerAction($handler, $params) {
        $parts = explode('@', $handler);
        
        if (count($parts) !== 2) {
            throw new Exception("Handler inválido: $handler");
        }
        
        $controllerName = $parts[0];
        $actionName = $parts[1];
        
        $controllerFile = CONTROLLERS_PATH . "/{$controllerName}.php";
        
        if (!file_exists($controllerFile)) {
            throw new Exception("Controller não encontrado: $controllerName");
        }
        
        require_once $controllerFile;
        
        if (!class_exists($controllerName)) {
            throw new Exception("Classe do controller não encontrada: $controllerName");
        }
        
        $controller = new $controllerName();
        
        if (!method_exists($controller, $actionName)) {
            throw new Exception("Método não encontrado: {$controllerName}@{$actionName}");
        }
        
        call_user_func_array([$controller, $actionName], [$params]);
    }
    
    // Definir handler para rota não encontrada
    public function setNotFoundHandler($handler) {
        $this->notFoundHandler = $handler;
    }
    
    // Redirecionar
    public static function redirect($url, $code = 302) {
        header("Location: $url", true, $code);
        exit;
    }
    
    // Obter URL atual
    public static function getCurrentUrl() {
        return $_SERVER['REQUEST_URI'];
    }
    
    // Gerar URL
    public static function url($path) {
        $baseUrl = rtrim(APP_URL, '/');
        return $baseUrl . '/' . ltrim($path, '/');
    }
}
