
<?php

require_once ROOT_PATH . '/src/models/User.php';

class AuthService {
    
    private $sessionTimeout = SESSION_TIMEOUT;
    
    public function __construct() {
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
    }
    
    // Fazer login
    public function login($email, $password, $remember = false) {
        $user = new User();
        $foundUser = $user->findByEmail($email);
        
        if ($foundUser && $foundUser->verifyPassword($password)) {
            // Regenerar ID da sessão para segurança
            session_regenerate_id(true);
            
            // Definir dados da sessão
            $_SESSION['user_id'] = $foundUser->id;
            $_SESSION['user_name'] = $foundUser->nome;
            $_SESSION['user_email'] = $foundUser->email;
            $_SESSION['user_type'] = $foundUser->tipo;
            $_SESSION['login_time'] = time();
            $_SESSION['last_activity'] = time();
            
            // Cookie "lembrar-me" (opcional)
            if ($remember) {
                $token = bin2hex(random_bytes(32));
                setcookie('remember_token', $token, time() + (30 * 24 * 60 * 60), '/', '', true, true);
                // Aqui você salvaria o token no banco para verificação futura
            }
            
            // Log da ação
            logMessage("Login realizado: {$foundUser->email}");
            
            return [
                'success' => true,
                'user' => $foundUser
            ];
        }
        
        // Log da tentativa de login inválida
        logMessage("Tentativa de login inválida: {$email}", 'WARNING');
        
        return [
            'success' => false,
            'message' => 'Email ou senha inválidos'
        ];
    }
    
    // Fazer logout
    public function logout() {
        if ($this->isLoggedIn()) {
            $email = $_SESSION['user_email'] ?? 'desconhecido';
            
            // Remover cookie remember-me
            if (isset($_COOKIE['remember_token'])) {
                setcookie('remember_token', '', time() - 3600, '/', '', true, true);
            }
            
            // Destruir sessão
            session_destroy();
            
            // Log da ação
            logMessage("Logout realizado: {$email}");
            
            return true;
        }
        
        return false;
    }
    
    // Verificar se usuário está logado
    public function isLoggedIn() {
        if (!isset($_SESSION['user_id'])) {
            return false;
        }
        
        // Verificar timeout da sessão
        if (isset($_SESSION['last_activity']) && 
            (time() - $_SESSION['last_activity'] > $this->sessionTimeout)) {
            $this->logout();
            return false;
        }
        
        // Atualizar última atividade
        $_SESSION['last_activity'] = time();
        
        return true;
    }
    
    // Obter usuário atual
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        $user = new User();
        return $user->findById($_SESSION['user_id']);
    }
    
    // Verificar permissão
    public function hasPermission($requiredType) {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        $userType = $_SESSION['user_type'] ?? '';
        
        // Admin tem todas as permissões
        if ($userType === 'admin') {
            return true;
        }
        
        // Verificar permissões específicas
        $permissions = [
            'admin' => ['admin'],
            'rh' => ['admin', 'rh'],
            'recrutador' => ['admin', 'rh', 'recrutador']
        ];
        
        return in_array($userType, $permissions[$requiredType] ?? []);
    }
    
    // Registrar usuário
    public function register($userData) {
        $user = new User();
        
        // Verificar se email já existe
        if ($user->emailExists($userData['email'])) {
            return [
                'success' => false,
                'message' => 'Este email já está em uso'
            ];
        }
        
        // Validar dados
        $validation = $this->validateUserData($userData);
        if (!$validation['valid']) {
            return [
                'success' => false,
                'message' => $validation['message']
            ];
        }
        
        // Mapear dados
        $user->nome = $userData['nome'];
        $user->email = $userData['email'];
        $user->senha = $userData['senha'];
        $user->tipo = $userData['tipo'] ?? 'recrutador';
        $user->ativo = 1;
        $user->telefone = $userData['telefone'] ?? '';
        $user->departamento = $userData['departamento'] ?? '';
        
        if ($user->create()) {
            logMessage("Usuário registrado: {$user->email}");
            
            return [
                'success' => true,
                'message' => 'Usuário criado com sucesso',
                'user_id' => $user->id
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Erro ao criar usuário'
        ];
    }
    
    // Validar dados do usuário
    private function validateUserData($data) {
        if (empty($data['nome'])) {
            return ['valid' => false, 'message' => 'Nome é obrigatório'];
        }
        
        if (empty($data['email']) || !isValidEmail($data['email'])) {
            return ['valid' => false, 'message' => 'Email válido é obrigatório'];
        }
        
        if (empty($data['senha']) || strlen($data['senha']) < PASSWORD_MIN_LENGTH) {
            return ['valid' => false, 'message' => 'Senha deve ter pelo menos ' . PASSWORD_MIN_LENGTH . ' caracteres'];
        }
        
        $allowedTypes = ['admin', 'rh', 'recrutador'];
        if (isset($data['tipo']) && !in_array($data['tipo'], $allowedTypes)) {
            return ['valid' => false, 'message' => 'Tipo de usuário inválido'];
        }
        
        return ['valid' => true];
    }
    
    // Alterar senha
    public function changePassword($currentPassword, $newPassword) {
        $user = $this->getCurrentUser();
        
        if (!$user) {
            return [
                'success' => false,
                'message' => 'Usuário não encontrado'
            ];
        }
        
        if (!$user->verifyPassword($currentPassword)) {
            return [
                'success' => false,
                'message' => 'Senha atual incorreta'
            ];
        }
        
        if (strlen($newPassword) < PASSWORD_MIN_LENGTH) {
            return [
                'success' => false,
                'message' => 'Nova senha deve ter pelo menos ' . PASSWORD_MIN_LENGTH . ' caracteres'
            ];
        }
        
        if ($user->updatePassword($newPassword)) {
            logMessage("Senha alterada: {$user->email}");
            
            return [
                'success' => true,
                'message' => 'Senha alterada com sucesso'
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Erro ao alterar senha'
        ];
    }
    
    // Obter informações da sessão
    public function getSessionInfo() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'user_id' => $_SESSION['user_id'],
            'user_name' => $_SESSION['user_name'],
            'user_email' => $_SESSION['user_email'],
            'user_type' => $_SESSION['user_type'],
            'login_time' => $_SESSION['login_time'],
            'last_activity' => $_SESSION['last_activity'],
            'time_remaining' => $this->sessionTimeout - (time() - $_SESSION['last_activity'])
        ];
    }
}
