
<?php

require_once ROOT_PATH . '/config/database.php';

class User {
    private $db;
    private $table = 'usuarios';
    
    public $id;
    public $nome;
    public $email;
    public $senha;
    public $tipo;
    public $ativo;
    public $foto;
    public $telefone;
    public $departamento;
    public $data_criacao;
    public $data_atualizacao;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    // Criar usuário
    public function create() {
        $sql = "INSERT INTO {$this->table} 
                (nome, email, senha, tipo, ativo, foto, telefone, departamento) 
                VALUES (:nome, :email, :senha, :tipo, :ativo, :foto, :telefone, :departamento)";
                
        $stmt = $this->db->prepare($sql);
        
        // Hash da senha
        $hashedPassword = password_hash($this->senha, PASSWORD_DEFAULT);
        
        $stmt->bindParam(':nome', $this->nome);
        $stmt->bindParam(':email', $this->email);
        $stmt->bindParam(':senha', $hashedPassword);
        $stmt->bindParam(':tipo', $this->tipo);
        $stmt->bindParam(':ativo', $this->ativo);
        $stmt->bindParam(':foto', $this->foto);
        $stmt->bindParam(':telefone', $this->telefone);
        $stmt->bindParam(':departamento', $this->departamento);
        
        if ($stmt->execute()) {
            $this->id = $this->db->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    // Buscar usuário por ID
    public function findById($id) {
        $sql = "SELECT * FROM {$this->table} WHERE id = :id AND ativo = 1";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $result = $stmt->fetch();
        if ($result) {
            $this->mapFromArray($result);
            return $this;
        }
        
        return false;
    }
    
    // Buscar usuário por email
    public function findByEmail($email) {
        $sql = "SELECT * FROM {$this->table} WHERE email = :email AND ativo = 1";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        
        $result = $stmt->fetch();
        if ($result) {
            $this->mapFromArray($result);
            return $this;
        }
        
        return false;
    }
    
    // Atualizar usuário
    public function update() {
        $sql = "UPDATE {$this->table} SET 
                nome = :nome, 
                email = :email,
                tipo = :tipo,
                ativo = :ativo,
                foto = :foto,
                telefone = :telefone,
                departamento = :departamento
                WHERE id = :id";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome', $this->nome);
        $stmt->bindParam(':email', $this->email);
        $stmt->bindParam(':tipo', $this->tipo);
        $stmt->bindParam(':ativo', $this->ativo);
        $stmt->bindParam(':foto', $this->foto);
        $stmt->bindParam(':telefone', $this->telefone);
        $stmt->bindParam(':departamento', $this->departamento);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Atualizar senha
    public function updatePassword($newPassword) {
        $sql = "UPDATE {$this->table} SET senha = :senha WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt->bindParam(':senha', $hashedPassword);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Verificar senha
    public function verifyPassword($password) {
        return password_verify($password, $this->senha);
    }
    
    // Listar usuários com paginação
    public function getAll($page = 1, $limit = ITEMS_PER_PAGE, $search = '') {
        $offset = ($page - 1) * $limit;
        
        $whereClause = "WHERE ativo = 1";
        $params = [];
        
        if (!empty($search)) {
            $whereClause .= " AND (nome LIKE :search OR email LIKE :search OR departamento LIKE :search)";
            $params[':search'] = "%{$search}%";
        }
        
        $sql = "SELECT * FROM {$this->table} 
                {$whereClause} 
                ORDER BY nome ASC 
                LIMIT :limit OFFSET :offset";
                
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Contar usuários
    public function count($search = '') {
        $whereClause = "WHERE ativo = 1";
        $params = [];
        
        if (!empty($search)) {
            $whereClause .= " AND (nome LIKE :search OR email LIKE :search OR departamento LIKE :search)";
            $params[':search'] = "%{$search}%";
        }
        
        $sql = "SELECT COUNT(*) as total FROM {$this->table} {$whereClause}";
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        
        return $result['total'] ?? 0;
    }
    
    // Desativar usuário (soft delete)
    public function delete() {
        $sql = "UPDATE {$this->table} SET ativo = 0 WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Verificar se email já existe
    public function emailExists($email, $excludeId = null) {
        $sql = "SELECT id FROM {$this->table} WHERE email = :email";
        $params = [':email' => $email];
        
        if ($excludeId) {
            $sql .= " AND id != :exclude_id";
            $params[':exclude_id'] = $excludeId;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetch() !== false;
    }
    
    // Mapear array para propriedades
    private function mapFromArray($data) {
        $this->id = $data['id'];
        $this->nome = $data['nome'];
        $this->email = $data['email'];
        $this->senha = $data['senha'];
        $this->tipo = $data['tipo'];
        $this->ativo = $data['ativo'];
        $this->foto = $data['foto'];
        $this->telefone = $data['telefone'];
        $this->departamento = $data['departamento'];
        $this->data_criacao = $data['data_criacao'];
        $this->data_atualizacao = $data['data_atualizacao'];
    }
    
    // Obter estatísticas de usuários
    public function getStats() {
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN tipo = 'admin' THEN 1 ELSE 0 END) as admins,
                    SUM(CASE WHEN tipo = 'rh' THEN 1 ELSE 0 END) as rh,
                    SUM(CASE WHEN tipo = 'recrutador' THEN 1 ELSE 0 END) as recrutadores,
                    SUM(CASE WHEN ativo = 1 THEN 1 ELSE 0 END) as ativos
                FROM {$this->table}";
                
        $stmt = $this->db->prepare($sql);
        $stmt->execute();
        
        return $stmt->fetch();
    }
}
