
<?php

require_once ROOT_PATH . '/config/database.php';

class Tag {
    private $db;
    private $table = 'etiquetas';
    
    public $id;
    public $nome;
    public $cor;
    public $usuario_id;
    public $data_criacao;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    // Criar etiqueta
    public function create() {
        $sql = "INSERT INTO {$this->table} (nome, cor, usuario_id) 
                VALUES (:nome, :cor, :usuario_id)";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome', $this->nome);
        $stmt->bindParam(':cor', $this->cor);
        $stmt->bindParam(':usuario_id', $this->usuario_id);
        
        if ($stmt->execute()) {
            $this->id = $this->db->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    // Buscar etiqueta por ID
    public function findById($id) {
        $sql = "SELECT e.*, u.nome as usuario_nome 
                FROM {$this->table} e
                LEFT JOIN usuarios u ON e.usuario_id = u.id
                WHERE e.id = :id";
                
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $result = $stmt->fetch();
        if ($result) {
            $this->mapFromArray($result);
            return $this;
        }
        
        return false;
    }
    
    // Atualizar etiqueta
    public function update() {
        $sql = "UPDATE {$this->table} SET
                nome = :nome,
                cor = :cor
                WHERE id = :id";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome', $this->nome);
        $stmt->bindParam(':cor', $this->cor);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Listar etiquetas
    public function getAll($userId = null, $search = '') {
        $whereConditions = [];
        $params = [];
        
        if ($userId) {
            $whereConditions[] = "e.usuario_id = :usuario_id";
            $params[':usuario_id'] = $userId;
        }
        
        if (!empty($search)) {
            $whereConditions[] = "e.nome LIKE :search";
            $params[':search'] = "%{$search}%";
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $sql = "SELECT e.*, u.nome as usuario_nome,
                (SELECT COUNT(*) FROM curriculo_etiquetas WHERE etiqueta_id = e.id) as total_curriculos
                FROM {$this->table} e
                LEFT JOIN usuarios u ON e.usuario_id = u.id
                {$whereClause}
                ORDER BY e.nome ASC";
                
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Excluir etiqueta
    public function delete() {
        // Primeiro, remover relacionamentos
        $sql = "DELETE FROM curriculo_etiquetas WHERE etiqueta_id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $this->id);
        $stmt->execute();
        
        // Depois, excluir a etiqueta
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Verificar se nome já existe para o usuário
    public function nameExists($nome, $userId, $excludeId = null) {
        $sql = "SELECT id FROM {$this->table} WHERE nome = :nome AND usuario_id = :usuario_id";
        $params = [':nome' => $nome, ':usuario_id' => $userId];
        
        if ($excludeId) {
            $sql .= " AND id != :exclude_id";
            $params[':exclude_id'] = $excludeId;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetch() !== false;
    }
    
    // Mapear array para propriedades
    private function mapFromArray($data) {
        $this->id = $data['id'];
        $this->nome = $data['nome'];
        $this->cor = $data['cor'];
        $this->usuario_id = $data['usuario_id'];
        $this->data_criacao = $data['data_criacao'];
    }
}
