
<?php

require_once ROOT_PATH . '/config/database.php';

class Group {
    private $db;
    private $table = 'grupos';
    
    public $id;
    public $nome;
    public $descricao;
    public $cor;
    public $usuario_id;
    public $data_criacao;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    // Criar grupo
    public function create() {
        $sql = "INSERT INTO {$this->table} (nome, descricao, cor, usuario_id) 
                VALUES (:nome, :descricao, :cor, :usuario_id)";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome', $this->nome);
        $stmt->bindParam(':descricao', $this->descricao);
        $stmt->bindParam(':cor', $this->cor);
        $stmt->bindParam(':usuario_id', $this->usuario_id);
        
        if ($stmt->execute()) {
            $this->id = $this->db->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    // Buscar grupo por ID
    public function findById($id) {
        $sql = "SELECT g.*, u.nome as usuario_nome 
                FROM {$this->table} g
                LEFT JOIN usuarios u ON g.usuario_id = u.id
                WHERE g.id = :id";
                
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $result = $stmt->fetch();
        if ($result) {
            $this->mapFromArray($result);
            return $this;
        }
        
        return false;
    }
    
    // Atualizar grupo
    public function update() {
        $sql = "UPDATE {$this->table} SET
                nome = :nome,
                descricao = :descricao,
                cor = :cor
                WHERE id = :id";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome', $this->nome);
        $stmt->bindParam(':descricao', $this->descricao);
        $stmt->bindParam(':cor', $this->cor);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Listar grupos
    public function getAll($userId = null, $search = '') {
        $whereConditions = [];
        $params = [];
        
        if ($userId) {
            $whereConditions[] = "usuario_id = :usuario_id";
            $params[':usuario_id'] = $userId;
        }
        
        if (!empty($search)) {
            $whereConditions[] = "(nome LIKE :search OR descricao LIKE :search)";
            $params[':search'] = "%{$search}%";
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $sql = "SELECT g.*, u.nome as usuario_nome,
                (SELECT COUNT(*) FROM curriculos WHERE grupo_id = g.id) as total_curriculos
                FROM {$this->table} g
                LEFT JOIN usuarios u ON g.usuario_id = u.id
                {$whereClause}
                ORDER BY g.nome ASC";
                
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Excluir grupo
    public function delete() {
        // Primeiro, remover referências dos currículos
        $sql = "UPDATE curriculos SET grupo_id = NULL WHERE grupo_id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $this->id);
        $stmt->execute();
        
        // Depois, excluir o grupo
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Verificar se nome já existe para o usuário
    public function nameExists($nome, $userId, $excludeId = null) {
        $sql = "SELECT id FROM {$this->table} WHERE nome = :nome AND usuario_id = :usuario_id";
        $params = [':nome' => $nome, ':usuario_id' => $userId];
        
        if ($excludeId) {
            $sql .= " AND id != :exclude_id";
            $params[':exclude_id'] = $excludeId;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetch() !== false;
    }
    
    // Obter estatísticas do grupo
    public function getStats() {
        $sql = "SELECT 
                    COUNT(c.id) as total_curriculos,
                    SUM(CASE WHEN c.status = 'novo' THEN 1 ELSE 0 END) as novos,
                    SUM(CASE WHEN c.status = 'aprovado' THEN 1 ELSE 0 END) as aprovados,
                    AVG(c.pontuacao) as pontuacao_media
                FROM curriculos c
                WHERE c.grupo_id = :grupo_id";
                
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':grupo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    // Mapear array para propriedades
    private function mapFromArray($data) {
        $this->id = $data['id'];
        $this->nome = $data['nome'];
        $this->descricao = $data['descricao'];
        $this->cor = $data['cor'];
        $this->usuario_id = $data['usuario_id'];
        $this->data_criacao = $data['data_criacao'];
    }
}
