
<?php

require_once ROOT_PATH . '/config/database.php';

class Curriculum {
    private $db;
    private $table = 'curriculos';
    
    public $id;
    public $nome_completo;
    public $email;
    public $telefone;
    public $celular;
    public $data_nascimento;
    public $endereco;
    public $cidade;
    public $estado;
    public $cep;
    public $objetivo;
    public $resumo;
    public $linkedin;
    public $github;
    public $website;
    public $foto;
    public $usuario_criador_id;
    public $grupo_id;
    public $status;
    public $pontuacao;
    public $observacoes;
    public $data_criacao;
    public $data_atualizacao;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    // Criar currículo
    public function create() {
        $sql = "INSERT INTO {$this->table} 
                (nome_completo, email, telefone, celular, data_nascimento, endereco, 
                 cidade, estado, cep, objetivo, resumo, linkedin, github, website, 
                 foto, usuario_criador_id, grupo_id, status, pontuacao, observacoes) 
                VALUES 
                (:nome_completo, :email, :telefone, :celular, :data_nascimento, :endereco,
                 :cidade, :estado, :cep, :objetivo, :resumo, :linkedin, :github, :website,
                 :foto, :usuario_criador_id, :grupo_id, :status, :pontuacao, :observacoes)";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome_completo', $this->nome_completo);
        $stmt->bindParam(':email', $this->email);
        $stmt->bindParam(':telefone', $this->telefone);
        $stmt->bindParam(':celular', $this->celular);
        $stmt->bindParam(':data_nascimento', $this->data_nascimento);
        $stmt->bindParam(':endereco', $this->endereco);
        $stmt->bindParam(':cidade', $this->cidade);
        $stmt->bindParam(':estado', $this->estado);
        $stmt->bindParam(':cep', $this->cep);
        $stmt->bindParam(':objetivo', $this->objetivo);
        $stmt->bindParam(':resumo', $this->resumo);
        $stmt->bindParam(':linkedin', $this->linkedin);
        $stmt->bindParam(':github', $this->github);
        $stmt->bindParam(':website', $this->website);
        $stmt->bindParam(':foto', $this->foto);
        $stmt->bindParam(':usuario_criador_id', $this->usuario_criador_id);
        $stmt->bindParam(':grupo_id', $this->grupo_id);
        $stmt->bindParam(':status', $this->status);
        $stmt->bindParam(':pontuacao', $this->pontuacao);
        $stmt->bindParam(':observacoes', $this->observacoes);
        
        if ($stmt->execute()) {
            $this->id = $this->db->lastInsertId();
            return true;
        }
        
        return false;
    }
    
    // Buscar currículo por ID
    public function findById($id) {
        $sql = "SELECT c.*, u.nome as usuario_criador, g.nome as grupo_nome 
                FROM {$this->table} c
                LEFT JOIN usuarios u ON c.usuario_criador_id = u.id
                LEFT JOIN grupos g ON c.grupo_id = g.id
                WHERE c.id = :id";
                
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $result = $stmt->fetch();
        if ($result) {
            $this->mapFromArray($result);
            return $this;
        }
        
        return false;
    }
    
    // Atualizar currículo
    public function update() {
        $sql = "UPDATE {$this->table} SET
                nome_completo = :nome_completo,
                email = :email,
                telefone = :telefone,
                celular = :celular,
                data_nascimento = :data_nascimento,
                endereco = :endereco,
                cidade = :cidade,
                estado = :estado,
                cep = :cep,
                objetivo = :objetivo,
                resumo = :resumo,
                linkedin = :linkedin,
                github = :github,
                website = :website,
                foto = :foto,
                grupo_id = :grupo_id,
                status = :status,
                pontuacao = :pontuacao,
                observacoes = :observacoes
                WHERE id = :id";
                
        $stmt = $this->db->prepare($sql);
        
        $stmt->bindParam(':nome_completo', $this->nome_completo);
        $stmt->bindParam(':email', $this->email);
        $stmt->bindParam(':telefone', $this->telefone);
        $stmt->bindParam(':celular', $this->celular);
        $stmt->bindParam(':data_nascimento', $this->data_nascimento);
        $stmt->bindParam(':endereco', $this->endereco);
        $stmt->bindParam(':cidade', $this->cidade);
        $stmt->bindParam(':estado', $this->estado);
        $stmt->bindParam(':cep', $this->cep);
        $stmt->bindParam(':objetivo', $this->objetivo);
        $stmt->bindParam(':resumo', $this->resumo);
        $stmt->bindParam(':linkedin', $this->linkedin);
        $stmt->bindParam(':github', $this->github);
        $stmt->bindParam(':website', $this->website);
        $stmt->bindParam(':foto', $this->foto);
        $stmt->bindParam(':grupo_id', $this->grupo_id);
        $stmt->bindParam(':status', $this->status);
        $stmt->bindParam(':pontuacao', $this->pontuacao);
        $stmt->bindParam(':observacoes', $this->observacoes);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Listar currículos com filtros e paginação
    public function getAll($filters = [], $page = 1, $limit = ITEMS_PER_PAGE) {
        $offset = ($page - 1) * $limit;
        $whereConditions = [];
        $params = [];
        
        // Filtros
        if (!empty($filters['search'])) {
            $whereConditions[] = "(c.nome_completo LIKE :search OR c.email LIKE :search OR c.telefone LIKE :search)";
            $params[':search'] = "%{$filters['search']}%";
        }
        
        if (!empty($filters['status'])) {
            $whereConditions[] = "c.status = :status";
            $params[':status'] = $filters['status'];
        }
        
        if (!empty($filters['grupo_id'])) {
            $whereConditions[] = "c.grupo_id = :grupo_id";
            $params[':grupo_id'] = $filters['grupo_id'];
        }
        
        if (!empty($filters['cidade'])) {
            $whereConditions[] = "c.cidade LIKE :cidade";
            $params[':cidade'] = "%{$filters['cidade']}%";
        }
        
        if (!empty($filters['usuario_criador_id'])) {
            $whereConditions[] = "c.usuario_criador_id = :usuario_criador_id";
            $params[':usuario_criador_id'] = $filters['usuario_criador_id'];
        }
        
        if (!empty($filters['pontuacao_min'])) {
            $whereConditions[] = "c.pontuacao >= :pontuacao_min";
            $params[':pontuacao_min'] = $filters['pontuacao_min'];
        }
        
        if (!empty($filters['pontuacao_max'])) {
            $whereConditions[] = "c.pontuacao <= :pontuacao_max";
            $params[':pontuacao_max'] = $filters['pontuacao_max'];
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $orderBy = "ORDER BY ";
        switch ($filters['sort'] ?? 'data_criacao') {
            case 'nome':
                $orderBy .= "c.nome_completo ASC";
                break;
            case 'pontuacao':
                $orderBy .= "c.pontuacao DESC";
                break;
            case 'status':
                $orderBy .= "c.status ASC";
                break;
            default:
                $orderBy .= "c.data_criacao DESC";
        }
        
        $sql = "SELECT c.*, u.nome as usuario_criador, g.nome as grupo_nome, g.cor as grupo_cor
                FROM {$this->table} c
                LEFT JOIN usuarios u ON c.usuario_criador_id = u.id
                LEFT JOIN grupos g ON c.grupo_id = g.id
                {$whereClause}
                {$orderBy}
                LIMIT :limit OFFSET :offset";
                
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Contar currículos
    public function count($filters = []) {
        $whereConditions = [];
        $params = [];
        
        // Aplicar os mesmos filtros da busca
        if (!empty($filters['search'])) {
            $whereConditions[] = "(c.nome_completo LIKE :search OR c.email LIKE :search OR c.telefone LIKE :search)";
            $params[':search'] = "%{$filters['search']}%";
        }
        
        if (!empty($filters['status'])) {
            $whereConditions[] = "c.status = :status";
            $params[':status'] = $filters['status'];
        }
        
        if (!empty($filters['grupo_id'])) {
            $whereConditions[] = "c.grupo_id = :grupo_id";
            $params[':grupo_id'] = $filters['grupo_id'];
        }
        
        if (!empty($filters['cidade'])) {
            $whereConditions[] = "c.cidade LIKE :cidade";
            $params[':cidade'] = "%{$filters['cidade']}%";
        }
        
        if (!empty($filters['usuario_criador_id'])) {
            $whereConditions[] = "c.usuario_criador_id = :usuario_criador_id";
            $params[':usuario_criador_id'] = $filters['usuario_criador_id'];
        }
        
        if (!empty($filters['pontuacao_min'])) {
            $whereConditions[] = "c.pontuacao >= :pontuacao_min";
            $params[':pontuacao_min'] = $filters['pontuacao_min'];
        }
        
        if (!empty($filters['pontuacao_max'])) {
            $whereConditions[] = "c.pontuacao <= :pontuacao_max";
            $params[':pontuacao_max'] = $filters['pontuacao_max'];
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $sql = "SELECT COUNT(*) as total 
                FROM {$this->table} c
                LEFT JOIN usuarios u ON c.usuario_criador_id = u.id
                LEFT JOIN grupos g ON c.grupo_id = g.id
                {$whereClause}";
                
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        
        return $result['total'] ?? 0;
    }
    
    // Excluir currículo
    public function delete() {
        $sql = "DELETE FROM {$this->table} WHERE id = :id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':id', $this->id);
        
        return $stmt->execute();
    }
    
    // Obter experiências do currículo
    public function getExperiencias() {
        $sql = "SELECT * FROM experiencias WHERE curriculo_id = :curriculo_id ORDER BY ordem_exibicao, data_inicio DESC";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Obter formação do currículo
    public function getFormacao() {
        $sql = "SELECT * FROM formacao WHERE curriculo_id = :curriculo_id ORDER BY ordem_exibicao, data_inicio DESC";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Obter habilidades do currículo
    public function getHabilidades() {
        $sql = "SELECT * FROM habilidades WHERE curriculo_id = :curriculo_id ORDER BY tipo, ordem_exibicao";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Obter idiomas do currículo
    public function getIdiomas() {
        $sql = "SELECT * FROM idiomas WHERE curriculo_id = :curriculo_id ORDER BY idioma";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Obter arquivos do currículo
    public function getArquivos() {
        $sql = "SELECT * FROM arquivos WHERE curriculo_id = :curriculo_id ORDER BY tipo, data_upload DESC";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Obter etiquetas do currículo
    public function getEtiquetas() {
        $sql = "SELECT e.* FROM etiquetas e
                INNER JOIN curriculo_etiquetas ce ON e.id = ce.etiqueta_id
                WHERE ce.curriculo_id = :curriculo_id
                ORDER BY e.nome";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->execute();
        
        return $stmt->fetchAll();
    }
    
    // Adicionar etiqueta ao currículo
    public function addEtiqueta($etiquetaId) {
        $sql = "INSERT IGNORE INTO curriculo_etiquetas (curriculo_id, etiqueta_id) VALUES (:curriculo_id, :etiqueta_id)";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->bindParam(':etiqueta_id', $etiquetaId);
        
        return $stmt->execute();
    }
    
    // Remover etiqueta do currículo
    public function removeEtiqueta($etiquetaId) {
        $sql = "DELETE FROM curriculo_etiquetas WHERE curriculo_id = :curriculo_id AND etiqueta_id = :etiqueta_id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $this->id);
        $stmt->bindParam(':etiqueta_id', $etiquetaId);
        
        return $stmt->execute();
    }
    
    // Obter estatísticas
    public function getStats($filters = []) {
        $whereConditions = [];
        $params = [];
        
        if (!empty($filters['usuario_criador_id'])) {
            $whereConditions[] = "usuario_criador_id = :usuario_criador_id";
            $params[':usuario_criador_id'] = $filters['usuario_criador_id'];
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'novo' THEN 1 ELSE 0 END) as novos,
                    SUM(CASE WHEN status = 'triagem' THEN 1 ELSE 0 END) as triagem,
                    SUM(CASE WHEN status = 'aprovado' THEN 1 ELSE 0 END) as aprovados,
                    SUM(CASE WHEN status = 'rejeitado' THEN 1 ELSE 0 END) as rejeitados,
                    SUM(CASE WHEN status = 'entrevista' THEN 1 ELSE 0 END) as entrevistas,
                    SUM(CASE WHEN status = 'contratado' THEN 1 ELSE 0 END) as contratados,
                    AVG(pontuacao) as pontuacao_media,
                    MAX(pontuacao) as pontuacao_maxima
                FROM {$this->table}
                {$whereClause}";
                
        $stmt = $this->db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    // Mapear array para propriedades
    private function mapFromArray($data) {
        $this->id = $data['id'];
        $this->nome_completo = $data['nome_completo'];
        $this->email = $data['email'];
        $this->telefone = $data['telefone'];
        $this->celular = $data['celular'];
        $this->data_nascimento = $data['data_nascimento'];
        $this->endereco = $data['endereco'];
        $this->cidade = $data['cidade'];
        $this->estado = $data['estado'];
        $this->cep = $data['cep'];
        $this->objetivo = $data['objetivo'];
        $this->resumo = $data['resumo'];
        $this->linkedin = $data['linkedin'];
        $this->github = $data['github'];
        $this->website = $data['website'];
        $this->foto = $data['foto'];
        $this->usuario_criador_id = $data['usuario_criador_id'];
        $this->grupo_id = $data['grupo_id'];
        $this->status = $data['status'];
        $this->pontuacao = $data['pontuacao'];
        $this->observacoes = $data['observacoes'];
        $this->data_criacao = $data['data_criacao'];
        $this->data_atualizacao = $data['data_atualizacao'];
    }
}
