
<?php

require_once ROOT_PATH . '/src/models/User.php';
require_once ROOT_PATH . '/src/services/AuthService.php';

class UserController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Listar usuários
    public function index() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin e RH)
        if (!in_array($currentUser->tipo, ['admin', 'rh'])) {
            $_SESSION['error'] = 'Você não tem permissão para acessar esta página';
            header('Location: /dashboard');
            exit;
        }
        
        $user = new User();
        
        // Parâmetros de busca
        $page = (int)($_GET['page'] ?? 1);
        $search = sanitize($_GET['search'] ?? '');
        
        // Obter usuários
        $users = $user->getAll($page, ITEMS_PER_PAGE, $search);
        $totalUsers = $user->count($search);
        $totalPages = ceil($totalUsers / ITEMS_PER_PAGE);
        
        // Estatísticas
        $stats = $user->getStats();
        
        // Paginação
        $pagination = [
            'current' => $page,
            'total' => $totalPages,
            'items' => $totalUsers,
            'per_page' => ITEMS_PER_PAGE
        ];
        
        $title = 'Usuários';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Usuários'];
        
        require VIEWS_PATH . '/usuarios/index.php';
    }
    
    // Visualizar usuário
    public function show($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if (!in_array($currentUser->tipo, ['admin', 'rh'])) {
            $_SESSION['error'] = 'Você não tem permissão para acessar esta página';
            header('Location: /dashboard');
            exit;
        }
        
        $id = (int)$params['id'];
        $user = new User();
        
        $usuario = $user->findById($id);
        if (!$usuario) {
            $_SESSION['error'] = 'Usuário não encontrado';
            header('Location: /usuarios');
            exit;
        }
        
        $title = 'Usuário - ' . $usuario->nome;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Usuários' => '/usuarios', 'Visualizar'];
        
        require VIEWS_PATH . '/usuarios/show.php';
    }
    
    // Formulário de criação (apenas admin)
    public function create() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin)
        if ($currentUser->tipo !== 'admin') {
            $_SESSION['error'] = 'Você não tem permissão para acessar esta página';
            header('Location: /dashboard');
            exit;
        }
        
        // Dados do formulário (em caso de erro)
        $formData = $_SESSION['form_data'] ?? [];
        unset($_SESSION['form_data']);
        
        $title = 'Novo Usuário';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Usuários' => '/usuarios', 'Novo'];
        
        require VIEWS_PATH . '/usuarios/create.php';
    }
    
    // Salvar usuário (apenas admin)
    public function store() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin)
        if ($currentUser->tipo !== 'admin') {
            $_SESSION['error'] = 'Você não tem permissão para realizar esta ação';
            header('Location: /dashboard');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /usuarios/create');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /usuarios/create');
            exit;
        }
        
        $userData = [
            'nome' => sanitize($_POST['nome'] ?? ''),
            'email' => sanitize($_POST['email'] ?? ''),
            'senha' => $_POST['senha'] ?? '',
            'senha_confirmacao' => $_POST['senha_confirmacao'] ?? '',
            'tipo' => sanitize($_POST['tipo'] ?? 'recrutador'),
            'telefone' => sanitize($_POST['telefone'] ?? ''),
            'departamento' => sanitize($_POST['departamento'] ?? '')
        ];
        
        $result = $this->authService->register($userData);
        
        if ($result['success']) {
            $_SESSION['success'] = 'Usuário criado com sucesso!';
            header("Location: /usuarios/{$result['user_id']}");
            exit;
        } else {
            $_SESSION['error'] = $result['message'];
            $_SESSION['form_data'] = $userData;
            header('Location: /usuarios/create');
            exit;
        }
    }
    
    // Formulário de edição (apenas admin)
    public function edit($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin)
        if ($currentUser->tipo !== 'admin') {
            $_SESSION['error'] = 'Você não tem permissão para acessar esta página';
            header('Location: /dashboard');
            exit;
        }
        
        $id = (int)$params['id'];
        $user = new User();
        
        $usuario = $user->findById($id);
        if (!$usuario) {
            $_SESSION['error'] = 'Usuário não encontrado';
            header('Location: /usuarios');
            exit;
        }
        
        // Dados do formulário
        $formData = $_SESSION['form_data'] ?? [
            'nome' => $usuario->nome,
            'email' => $usuario->email,
            'tipo' => $usuario->tipo,
            'telefone' => $usuario->telefone,
            'departamento' => $usuario->departamento
        ];
        unset($_SESSION['form_data']);
        
        $title = 'Editar Usuário - ' . $usuario->nome;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Usuários' => '/usuarios', 'Editar'];
        
        require VIEWS_PATH . '/usuarios/edit.php';
    }
    
    // Atualizar usuário (apenas admin)
    public function update($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin)
        if ($currentUser->tipo !== 'admin') {
            $_SESSION['error'] = 'Você não tem permissão para realizar esta ação';
            header('Location: /dashboard');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /usuarios');
            exit;
        }
        
        $id = (int)$params['id'];
        $user = new User();
        
        $usuario = $user->findById($id);
        if (!$usuario) {
            $_SESSION['error'] = 'Usuário não encontrado';
            header('Location: /usuarios');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header("Location: /usuarios/{$id}/edit");
            exit;
        }
        
        // Validar dados
        $errors = [];
        
        $nome = sanitize($_POST['nome'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        $tipo = sanitize($_POST['tipo'] ?? '');
        $telefone = sanitize($_POST['telefone'] ?? '');
        $departamento = sanitize($_POST['departamento'] ?? '');
        
        if (empty($nome)) {
            $errors[] = 'Nome é obrigatório';
        }
        
        if (empty($email) || !isValidEmail($email)) {
            $errors[] = 'Email válido é obrigatório';
        }
        
        // Verificar se email já existe (para outro usuário)
        if ($user->emailExists($email, $id)) {
            $errors[] = 'Este email já está em uso';
        }
        
        if (!in_array($tipo, ['admin', 'rh', 'recrutador'])) {
            $errors[] = 'Tipo de usuário inválido';
        }
        
        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            $_SESSION['form_data'] = $_POST;
            header("Location: /usuarios/{$id}/edit");
            exit;
        }
        
        // Atualizar dados
        $usuario->nome = $nome;
        $usuario->email = $email;
        $usuario->tipo = $tipo;
        $usuario->telefone = $telefone;
        $usuario->departamento = $departamento;
        
        if ($usuario->update()) {
            $_SESSION['success'] = 'Usuário atualizado com sucesso!';
            header("Location: /usuarios/{$id}");
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao atualizar usuário';
            $_SESSION['form_data'] = $_POST;
            header("Location: /usuarios/{$id}/edit");
            exit;
        }
    }
    
    // Excluir usuário (apenas admin)
    public function delete($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin)
        if ($currentUser->tipo !== 'admin') {
            $_SESSION['error'] = 'Você não tem permissão para realizar esta ação';
            header('Location: /dashboard');
            exit;
        }
        
        $id = (int)$params['id'];
        $user = new User();
        
        $usuario = $user->findById($id);
        if (!$usuario) {
            $_SESSION['error'] = 'Usuário não encontrado';
            header('Location: /usuarios');
            exit;
        }
        
        // Não permitir excluir a si mesmo
        if ($usuario->id == $currentUser->id) {
            $_SESSION['error'] = 'Você não pode excluir sua própria conta';
            header('Location: /usuarios');
            exit;
        }
        
        if ($usuario->delete()) {
            $_SESSION['success'] = 'Usuário excluído com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao excluir usuário';
        }
        
        header('Location: /usuarios');
        exit;
    }
}
