
<?php

require_once ROOT_PATH . '/src/models/Tag.php';
require_once ROOT_PATH . '/src/services/AuthService.php';

class TagController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Listar etiquetas
    public function index() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        $tag = new Tag();
        
        // Parâmetros de busca
        $search = sanitize($_GET['search'] ?? '');
        
        // Obter etiquetas (admin vê todas, outros apenas as próprias)
        $tags = $tag->getAll($currentUser->tipo === 'admin' ? null : $currentUser->id, $search);
        
        $title = 'Etiquetas';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Etiquetas'];
        
        require VIEWS_PATH . '/etiquetas/index.php';
    }
    
    // Formulário de criação
    public function create() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        // Dados do formulário (em caso de erro)
        $formData = $_SESSION['form_data'] ?? [];
        unset($_SESSION['form_data']);
        
        $title = 'Nova Etiqueta';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Etiquetas' => '/etiquetas', 'Nova'];
        
        require VIEWS_PATH . '/etiquetas/create.php';
    }
    
    // Salvar etiqueta
    public function store() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /etiquetas/create');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /etiquetas/create');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Validar dados
        $data = $this->validateTagData($_POST);
        
        if (isset($data['errors'])) {
            $_SESSION['error'] = implode('<br>', $data['errors']);
            $_SESSION['form_data'] = $_POST;
            header('Location: /etiquetas/create');
            exit;
        }
        
        // Criar etiqueta
        $tag = new Tag();
        $tag->nome = $data['nome'];
        $tag->cor = $data['cor'];
        $tag->usuario_id = $currentUser->id;
        
        if ($tag->create()) {
            $_SESSION['success'] = 'Etiqueta criada com sucesso!';
            header('Location: /etiquetas');
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao criar etiqueta';
            $_SESSION['form_data'] = $_POST;
            header('Location: /etiquetas/create');
            exit;
        }
    }
    
    // Formulário de edição
    public function edit($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $tag = new Tag();
        
        $etiqueta = $tag->findById($id);
        if (!$etiqueta) {
            $_SESSION['error'] = 'Etiqueta não encontrada';
            header('Location: /etiquetas');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $etiqueta->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para editar esta etiqueta';
            header('Location: /etiquetas');
            exit;
        }
        
        // Dados do formulário
        $formData = $_SESSION['form_data'] ?? [
            'nome' => $etiqueta->nome,
            'cor' => $etiqueta->cor
        ];
        unset($_SESSION['form_data']);
        
        $title = 'Editar Etiqueta - ' . $etiqueta->nome;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Etiquetas' => '/etiquetas', 'Editar'];
        
        require VIEWS_PATH . '/etiquetas/edit.php';
    }
    
    // Atualizar etiqueta
    public function update($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /etiquetas');
            exit;
        }
        
        $id = (int)$params['id'];
        $tag = new Tag();
        
        $etiqueta = $tag->findById($id);
        if (!$etiqueta) {
            $_SESSION['error'] = 'Etiqueta não encontrada';
            header('Location: /etiquetas');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $etiqueta->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para editar esta etiqueta';
            header('Location: /etiquetas');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header("Location: /etiquetas/{$id}/edit");
            exit;
        }
        
        // Validar dados
        $data = $this->validateTagData($_POST, $id);
        
        if (isset($data['errors'])) {
            $_SESSION['error'] = implode('<br>', $data['errors']);
            $_SESSION['form_data'] = $_POST;
            header("Location: /etiquetas/{$id}/edit");
            exit;
        }
        
        // Atualizar dados
        $etiqueta->nome = $data['nome'];
        $etiqueta->cor = $data['cor'];
        
        if ($etiqueta->update()) {
            $_SESSION['success'] = 'Etiqueta atualizada com sucesso!';
            header('Location: /etiquetas');
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao atualizar etiqueta';
            $_SESSION['form_data'] = $_POST;
            header("Location: /etiquetas/{$id}/edit");
            exit;
        }
    }
    
    // Excluir etiqueta
    public function delete($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $tag = new Tag();
        
        $etiqueta = $tag->findById($id);
        if (!$etiqueta) {
            $_SESSION['error'] = 'Etiqueta não encontrada';
            header('Location: /etiquetas');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $etiqueta->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para excluir esta etiqueta';
            header('Location: /etiquetas');
            exit;
        }
        
        if ($etiqueta->delete()) {
            $_SESSION['success'] = 'Etiqueta excluída com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao excluir etiqueta';
        }
        
        header('Location: /etiquetas');
        exit;
    }
    
    // Validar dados da etiqueta
    private function validateTagData($data, $excludeId = null) {
        $errors = [];
        $currentUser = $this->authService->getCurrentUser();
        
        // Nome obrigatório
        if (empty($data['nome'])) {
            $errors[] = 'Nome da etiqueta é obrigatório';
        }
        
        // Verificar se nome já existe para o usuário
        if (!empty($data['nome'])) {
            $tag = new Tag();
            if ($tag->nameExists($data['nome'], $currentUser->id, $excludeId)) {
                $errors[] = 'Já existe uma etiqueta com este nome';
            }
        }
        
        // Validar cor
        if (!empty($data['cor']) && !preg_match('/^#[0-9A-Fa-f]{6}$/', $data['cor'])) {
            $errors[] = 'Cor deve estar no formato hexadecimal (#000000)';
        }
        
        if (!empty($errors)) {
            return ['errors' => $errors];
        }
        
        return [
            'nome' => sanitize($data['nome']),
            'cor' => sanitize($data['cor'] ?? '#95a5a6')
        ];
    }
}
