
<?php

require_once ROOT_PATH . '/src/models/Group.php';
require_once ROOT_PATH . '/src/services/AuthService.php';

class GroupController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Listar grupos
    public function index() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        $group = new Group();
        
        // Parâmetros de busca
        $search = sanitize($_GET['search'] ?? '');
        
        // Obter grupos (admin vê todos, outros apenas os próprios)
        $groups = $group->getAll($currentUser->tipo === 'admin' ? null : $currentUser->id, $search);
        
        $title = 'Grupos';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Grupos'];
        
        require VIEWS_PATH . '/grupos/index.php';
    }
    
    // Visualizar grupo
    public function show($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $group = new Group();
        
        $grupo = $group->findById($id);
        if (!$grupo) {
            $_SESSION['error'] = 'Grupo não encontrado';
            header('Location: /grupos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $grupo->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para ver este grupo';
            header('Location: /grupos');
            exit;
        }
        
        // Obter estatísticas do grupo
        $stats = $grupo->getStats();
        
        // Obter currículos do grupo
        $curriculum = new Curriculum();
        $curriculos = $curriculum->getAll(['grupo_id' => $grupo->id], 1, 20);
        
        $title = 'Grupo - ' . $grupo->nome;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Grupos' => '/grupos', 'Visualizar'];
        
        require VIEWS_PATH . '/grupos/show.php';
    }
    
    // Formulário de criação
    public function create() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        // Dados do formulário (em caso de erro)
        $formData = $_SESSION['form_data'] ?? [];
        unset($_SESSION['form_data']);
        
        $title = 'Novo Grupo';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Grupos' => '/grupos', 'Novo'];
        
        require VIEWS_PATH . '/grupos/create.php';
    }
    
    // Salvar grupo
    public function store() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /grupos/create');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /grupos/create');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Validar dados
        $data = $this->validateGroupData($_POST);
        
        if (isset($data['errors'])) {
            $_SESSION['error'] = implode('<br>', $data['errors']);
            $_SESSION['form_data'] = $_POST;
            header('Location: /grupos/create');
            exit;
        }
        
        // Criar grupo
        $group = new Group();
        $group->nome = $data['nome'];
        $group->descricao = $data['descricao'];
        $group->cor = $data['cor'];
        $group->usuario_id = $currentUser->id;
        
        if ($group->create()) {
            $_SESSION['success'] = 'Grupo criado com sucesso!';
            header("Location: /grupos/{$group->id}");
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao criar grupo';
            $_SESSION['form_data'] = $_POST;
            header('Location: /grupos/create');
            exit;
        }
    }
    
    // Formulário de edição
    public function edit($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $group = new Group();
        
        $grupo = $group->findById($id);
        if (!$grupo) {
            $_SESSION['error'] = 'Grupo não encontrado';
            header('Location: /grupos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $grupo->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para editar este grupo';
            header('Location: /grupos');
            exit;
        }
        
        // Dados do formulário
        $formData = $_SESSION['form_data'] ?? [
            'nome' => $grupo->nome,
            'descricao' => $grupo->descricao,
            'cor' => $grupo->cor
        ];
        unset($_SESSION['form_data']);
        
        $title = 'Editar Grupo - ' . $grupo->nome;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Grupos' => '/grupos', 'Editar'];
        
        require VIEWS_PATH . '/grupos/edit.php';
    }
    
    // Atualizar grupo
    public function update($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /grupos');
            exit;
        }
        
        $id = (int)$params['id'];
        $group = new Group();
        
        $grupo = $group->findById($id);
        if (!$grupo) {
            $_SESSION['error'] = 'Grupo não encontrado';
            header('Location: /grupos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $grupo->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para editar este grupo';
            header('Location: /grupos');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header("Location: /grupos/{$id}/edit");
            exit;
        }
        
        // Validar dados
        $data = $this->validateGroupData($_POST, $id);
        
        if (isset($data['errors'])) {
            $_SESSION['error'] = implode('<br>', $data['errors']);
            $_SESSION['form_data'] = $_POST;
            header("Location: /grupos/{$id}/edit");
            exit;
        }
        
        // Atualizar dados
        $grupo->nome = $data['nome'];
        $grupo->descricao = $data['descricao'];
        $grupo->cor = $data['cor'];
        
        if ($grupo->update()) {
            $_SESSION['success'] = 'Grupo atualizado com sucesso!';
            header("Location: /grupos/{$grupo->id}");
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao atualizar grupo';
            $_SESSION['form_data'] = $_POST;
            header("Location: /grupos/{$id}/edit");
            exit;
        }
    }
    
    // Excluir grupo
    public function delete($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $group = new Group();
        
        $grupo = $group->findById($id);
        if (!$grupo) {
            $_SESSION['error'] = 'Grupo não encontrado';
            header('Location: /grupos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $grupo->usuario_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para excluir este grupo';
            header('Location: /grupos');
            exit;
        }
        
        if ($grupo->delete()) {
            $_SESSION['success'] = 'Grupo excluído com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao excluir grupo';
        }
        
        header('Location: /grupos');
        exit;
    }
    
    // Validar dados do grupo
    private function validateGroupData($data, $excludeId = null) {
        $errors = [];
        $currentUser = $this->authService->getCurrentUser();
        
        // Nome obrigatório
        if (empty($data['nome'])) {
            $errors[] = 'Nome do grupo é obrigatório';
        }
        
        // Verificar se nome já existe para o usuário
        if (!empty($data['nome'])) {
            $group = new Group();
            if ($group->nameExists($data['nome'], $currentUser->id, $excludeId)) {
                $errors[] = 'Já existe um grupo com este nome';
            }
        }
        
        // Validar cor
        if (!empty($data['cor']) && !preg_match('/^#[0-9A-Fa-f]{6}$/', $data['cor'])) {
            $errors[] = 'Cor deve estar no formato hexadecimal (#000000)';
        }
        
        if (!empty($errors)) {
            return ['errors' => $errors];
        }
        
        return [
            'nome' => sanitize($data['nome']),
            'descricao' => sanitize($data['descricao'] ?? ''),
            'cor' => sanitize($data['cor'] ?? '#3498db')
        ];
    }
}
