
<?php

require_once ROOT_PATH . '/src/services/AuthService.php';

class FileController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Upload de arquivos
    public function upload() {
        if (!$this->authService->isLoggedIn()) {
            http_response_code(401);
            echo json_encode(['error' => 'Não autorizado']);
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
            exit;
        }
        
        // Verificar se arquivo foi enviado
        if (!isset($_FILES['arquivo']) || $_FILES['arquivo']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => 'Erro no upload do arquivo']);
            exit;
        }
        
        $file = $_FILES['arquivo'];
        $curriculumId = (int)($_POST['curriculum_id'] ?? 0);
        $tipo = sanitize($_POST['tipo'] ?? 'curriculo');
        
        // Validar currículo
        if ($curriculumId <= 0) {
            http_response_code(400);
            echo json_encode(['error' => 'ID do currículo inválido']);
            exit;
        }
        
        // Verificar se o usuário tem permissão para o currículo
        $curriculum = new Curriculum();
        $cv = $curriculum->findById($curriculumId);
        
        if (!$cv) {
            http_response_code(404);
            echo json_encode(['error' => 'Currículo não encontrado']);
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        if ($currentUser->tipo !== 'admin' && $cv->usuario_criador_id != $currentUser->id) {
            http_response_code(403);
            echo json_encode(['error' => 'Sem permissão para este currículo']);
            exit;
        }
        
        // Validar arquivo
        $validation = $this->validateFile($file, $tipo);
        if (!$validation['valid']) {
            http_response_code(400);
            echo json_encode(['error' => $validation['message']]);
            exit;
        }
        
        // Salvar arquivo
        $result = $this->saveFile($file, $curriculumId, $tipo);
        
        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => 'Arquivo enviado com sucesso',
                'file' => $result['file']
            ]);
        } else {
            http_response_code(500);
            echo json_encode(['error' => $result['message']]);
        }
    }
    
    // Servir arquivos
    public function serve($params) {
        if (!$this->authService->isLoggedIn()) {
            http_response_code(401);
            exit;
        }
        
        $filename = $params['file'];
        
        // Validar nome do arquivo
        if (!$this->isValidFilename($filename)) {
            http_response_code(400);
            exit;
        }
        
        // Determinar tipo de arquivo pelo diretório
        $path = '';
        if (strpos($filename, 'curriculos/') === 0) {
            $path = UPLOAD_PATH . '/curriculos/' . basename($filename);
        } elseif (strpos($filename, 'fotos/') === 0) {
            $path = UPLOAD_PATH . '/fotos/' . basename($filename);
        } else {
            $path = UPLOAD_PATH . '/' . basename($filename);
        }
        
        // Verificar se arquivo existe
        if (!file_exists($path)) {
            http_response_code(404);
            exit;
        }
        
        // Obter informações do arquivo no banco
        $fileInfo = $this->getFileInfo($filename);
        
        if (!$fileInfo) {
            http_response_code(404);
            exit;
        }
        
        // Verificar permissão
        $currentUser = $this->authService->getCurrentUser();
        if ($currentUser->tipo !== 'admin' && $fileInfo['usuario_criador_id'] != $currentUser->id) {
            http_response_code(403);
            exit;
        }
        
        // Determinar content type
        $mimeType = $fileInfo['tipo_mime'] ?? $this->getMimeType($path);
        
        // Servir arquivo
        header('Content-Type: ' . $mimeType);
        header('Content-Length: ' . filesize($path));
        header('Content-Disposition: inline; filename="' . $fileInfo['nome_original'] . '"');
        header('Cache-Control: public, max-age=3600');
        
        readfile($path);
        exit;
    }
    
    // Validar arquivo
    private function validateFile($file, $tipo) {
        // Verificar tamanho
        if ($file['size'] > UPLOAD_MAX_SIZE) {
            return [
                'valid' => false,
                'message' => 'Arquivo muito grande. Máximo: ' . (UPLOAD_MAX_SIZE / 1024 / 1024) . 'MB'
            ];
        }
        
        // Verificar extensão
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        $allowedExtensions = [];
        switch ($tipo) {
            case 'curriculo':
                $allowedExtensions = ['pdf', 'doc', 'docx'];
                break;
            case 'foto':
                $allowedExtensions = ['jpg', 'jpeg', 'png'];
                break;
            case 'certificado':
                $allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png'];
                break;
            default:
                $allowedExtensions = ALLOWED_FILE_TYPES;
        }
        
        if (!in_array($extension, $allowedExtensions)) {
            return [
                'valid' => false,
                'message' => 'Tipo de arquivo não permitido. Permitidos: ' . implode(', ', $allowedExtensions)
            ];
        }
        
        // Verificar MIME type
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        $allowedMimes = [
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png'
        ];
        
        if (!in_array($mimeType, $allowedMimes) && !isset($allowedMimes[$extension])) {
            return [
                'valid' => false,
                'message' => 'Tipo MIME não permitido'
            ];
        }
        
        return ['valid' => true];
    }
    
    // Salvar arquivo
    private function saveFile($file, $curriculumId, $tipo) {
        try {
            // Criar diretório se não existir
            $uploadDir = UPLOAD_PATH . '/' . ($tipo === 'foto' ? 'fotos' : 'curriculos');
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            // Gerar nome único do arquivo
            $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
            $filename = uniqid() . '_' . time() . '.' . $extension;
            $filePath = $uploadDir . '/' . $filename;
            
            // Mover arquivo
            if (!move_uploaded_file($file['tmp_name'], $filePath)) {
                return [
                    'success' => false,
                    'message' => 'Erro ao salvar arquivo'
                ];
            }
            
            // Salvar no banco de dados
            $db = Database::getInstance()->getConnection();
            
            $sql = "INSERT INTO arquivos (curriculo_id, nome_original, nome_arquivo, tipo_mime, tamanho, tipo)
                    VALUES (:curriculo_id, :nome_original, :nome_arquivo, :tipo_mime, :tamanho, :tipo)";
                    
            $stmt = $db->prepare($sql);
            
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $filePath);
            finfo_close($finfo);
            
            $stmt->bindParam(':curriculo_id', $curriculumId);
            $stmt->bindParam(':nome_original', $file['name']);
            $stmt->bindParam(':nome_arquivo', ($tipo === 'foto' ? 'fotos/' : 'curriculos/') . $filename);
            $stmt->bindParam(':tipo_mime', $mimeType);
            $stmt->bindParam(':tamanho', $file['size']);
            $stmt->bindParam(':tipo', $tipo);
            
            if ($stmt->execute()) {
                return [
                    'success' => true,
                    'file' => [
                        'id' => $db->lastInsertId(),
                        'name' => $file['name'],
                        'filename' => ($tipo === 'foto' ? 'fotos/' : 'curriculos/') . $filename,
                        'size' => $file['size'],
                        'type' => $tipo
                    ]
                ];
            } else {
                // Remover arquivo se falhou ao salvar no banco
                unlink($filePath);
                return [
                    'success' => false,
                    'message' => 'Erro ao salvar informações do arquivo'
                ];
            }
        } catch (Exception $e) {
            logMessage("Erro no upload: " . $e->getMessage(), 'ERROR');
            return [
                'success' => false,
                'message' => 'Erro interno no servidor'
            ];
        }
    }
    
    // Verificar se nome do arquivo é válido
    private function isValidFilename($filename) {
        // Verificar se contém apenas caracteres seguros
        return preg_match('/^[a-zA-Z0-9\/_\-\.]+$/', $filename) && 
               strpos($filename, '..') === false;
    }
    
    // Obter informações do arquivo
    private function getFileInfo($filename) {
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT a.*, c.usuario_criador_id 
                FROM arquivos a
                INNER JOIN curriculos c ON a.curriculo_id = c.id
                WHERE a.nome_arquivo = :filename";
                
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':filename', $filename);
        $stmt->execute();
        
        return $stmt->fetch();
    }
    
    // Obter MIME type do arquivo
    private function getMimeType($path) {
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        
        $mimeTypes = [
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png'
        ];
        
        return $mimeTypes[$extension] ?? 'application/octet-stream';
    }
    
    // Excluir arquivo
    public function delete($params) {
        if (!$this->authService->isLoggedIn()) {
            http_response_code(401);
            echo json_encode(['error' => 'Não autorizado']);
            exit;
        }
        
        $id = (int)$params['id'];
        
        // Obter informações do arquivo
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT a.*, c.usuario_criador_id 
                FROM arquivos a
                INNER JOIN curriculos c ON a.curriculo_id = c.id
                WHERE a.id = :id";
                
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        $arquivo = $stmt->fetch();
        
        if (!$arquivo) {
            http_response_code(404);
            echo json_encode(['error' => 'Arquivo não encontrado']);
            exit;
        }
        
        // Verificar permissão
        $currentUser = $this->authService->getCurrentUser();
        if ($currentUser->tipo !== 'admin' && $arquivo['usuario_criador_id'] != $currentUser->id) {
            http_response_code(403);
            echo json_encode(['error' => 'Sem permissão']);
            exit;
        }
        
        // Excluir arquivo físico
        $filePath = UPLOAD_PATH . '/' . $arquivo['nome_arquivo'];
        if (file_exists($filePath)) {
            unlink($filePath);
        }
        
        // Excluir do banco
        $sql = "DELETE FROM arquivos WHERE id = :id";
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':id', $id);
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'message' => 'Arquivo excluído']);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao excluir arquivo']);
        }
    }
}
