
<?php

require_once ROOT_PATH . '/src/models/Curriculum.php';
require_once ROOT_PATH . '/src/models/User.php';
require_once ROOT_PATH . '/src/models/Group.php';
require_once ROOT_PATH . '/src/models/Tag.php';
require_once ROOT_PATH . '/src/services/AuthService.php';

class DashboardController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Dashboard principal
    public function index() {
        // Verificar autenticação
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Obter estatísticas gerais
        $curriculum = new Curriculum();
        $user = new User();
        $group = new Group();
        $tag = new Tag();
        
        // Filtrar por usuário se não for admin
        $filters = [];
        if ($currentUser->tipo !== 'admin') {
            $filters['usuario_criador_id'] = $currentUser->id;
        }
        
        // Estatísticas dos currículos
        $curriculumStats = $curriculum->getStats($filters);
        
        // Estatísticas dos usuários (apenas para admin e RH)
        $userStats = null;
        if (in_array($currentUser->tipo, ['admin', 'rh'])) {
            $userStats = $user->getStats();
        }
        
        // Últimos currículos adicionados
        $recentCurriculums = $curriculum->getAll($filters, 1, 5);
        
        // Grupos do usuário
        $groups = $group->getAll($currentUser->id);
        
        // Etiquetas do usuário
        $tags = $tag->getAll($currentUser->id);
        
        // Currículos por status (para gráfico)
        $statusData = [
            'labels' => [],
            'data' => []
        ];
        
        foreach (CURRICULUM_STATUS as $status => $label) {
            $statusData['labels'][] = $label;
            $statusData['data'][] = (int)($curriculumStats[$status] ?? 0);
        }
        
        // Currículos por mês (últimos 6 meses)
        $monthlyData = $this->getMonthlyStats($filters);
        
        $title = 'Dashboard';
        $breadcrumb = ['Dashboard'];
        
        require VIEWS_PATH . '/dashboard/index.php';
    }
    
    // Obter estatísticas mensais
    private function getMonthlyStats($filters = []) {
        $curriculum = new Curriculum();
        $db = Database::getInstance()->getConnection();
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($filters['usuario_criador_id'])) {
            $whereConditions[] = "usuario_criador_id = :usuario_criador_id";
            $params[':usuario_criador_id'] = $filters['usuario_criador_id'];
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $sql = "SELECT 
                    DATE_FORMAT(data_criacao, '%Y-%m') as mes,
                    COUNT(*) as total
                FROM curriculos
                {$whereClause}
                AND data_criacao >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
                GROUP BY DATE_FORMAT(data_criacao, '%Y-%m')
                ORDER BY mes ASC";
                
        $stmt = $db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $results = $stmt->fetchAll();
        
        // Preparar dados para os últimos 6 meses
        $monthlyData = [
            'labels' => [],
            'data' => []
        ];
        
        for ($i = 5; $i >= 0; $i--) {
            $month = date('Y-m', strtotime("-$i months"));
            $monthName = date('M/Y', strtotime("-$i months"));
            
            $monthlyData['labels'][] = $monthName;
            
            $found = false;
            foreach ($results as $result) {
                if ($result['mes'] === $month) {
                    $monthlyData['data'][] = (int)$result['total'];
                    $found = true;
                    break;
                }
            }
            
            if (!$found) {
                $monthlyData['data'][] = 0;
            }
        }
        
        return $monthlyData;
    }
    
    // Relatórios
    public function reports() {
        // Verificar autenticação
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão (apenas admin e RH podem ver relatórios completos)
        $canViewAllReports = in_array($currentUser->tipo, ['admin', 'rh']);
        
        $curriculum = new Curriculum();
        $user = new User();
        
        // Filtrar por usuário se não for admin/RH
        $filters = [];
        if (!$canViewAllReports) {
            $filters['usuario_criador_id'] = $currentUser->id;
        }
        
        // Estatísticas gerais
        $stats = $curriculum->getStats($filters);
        
        // Relatório por status
        $statusReport = [];
        foreach (CURRICULUM_STATUS as $status => $label) {
            $statusFilters = array_merge($filters, ['status' => $status]);
            $statusReport[] = [
                'status' => $status,
                'label' => $label,
                'count' => $curriculum->count($statusFilters),
                'percentage' => $stats['total'] > 0 ? round(($curriculum->count($statusFilters) / $stats['total']) * 100, 1) : 0
            ];
        }
        
        // Relatório por grupo
        $groupReport = [];
        $group = new Group();
        $groups = $group->getAll($canViewAllReports ? null : $currentUser->id);
        
        foreach ($groups as $groupData) {
            $groupFilters = array_merge($filters, ['grupo_id' => $groupData['id']]);
            $groupReport[] = [
                'grupo' => $groupData,
                'count' => $curriculum->count($groupFilters)
            ];
        }
        
        // Currículos sem grupo
        $noGroupFilters = array_merge($filters, ['grupo_id' => 'null']);
        $noGroupCount = $this->countCurriculumsWithoutGroup($filters);
        
        // Relatório mensal (últimos 12 meses)
        $monthlyReport = $this->getDetailedMonthlyStats($filters, 12);
        
        $title = 'Relatórios';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Relatórios'];
        
        require VIEWS_PATH . '/dashboard/reports.php';
    }
    
    // Contar currículos sem grupo
    private function countCurriculumsWithoutGroup($filters = []) {
        $db = Database::getInstance()->getConnection();
        
        $whereConditions = ['grupo_id IS NULL'];
        $params = [];
        
        if (!empty($filters['usuario_criador_id'])) {
            $whereConditions[] = "usuario_criador_id = :usuario_criador_id";
            $params[':usuario_criador_id'] = $filters['usuario_criador_id'];
        }
        
        $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
        
        $sql = "SELECT COUNT(*) as total FROM curriculos {$whereClause}";
        $stmt = $db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        
        return $result['total'] ?? 0;
    }
    
    // Obter estatísticas mensais detalhadas
    private function getDetailedMonthlyStats($filters = [], $months = 12) {
        $db = Database::getInstance()->getConnection();
        
        $whereConditions = [];
        $params = [];
        
        if (!empty($filters['usuario_criador_id'])) {
            $whereConditions[] = "usuario_criador_id = :usuario_criador_id";
            $params[':usuario_criador_id'] = $filters['usuario_criador_id'];
        }
        
        $whereClause = !empty($whereConditions) ? 'WHERE ' . implode(' AND ', $whereConditions) : '';
        
        $sql = "SELECT 
                    DATE_FORMAT(data_criacao, '%Y-%m') as mes,
                    COUNT(*) as total,
                    SUM(CASE WHEN status = 'novo' THEN 1 ELSE 0 END) as novos,
                    SUM(CASE WHEN status = 'aprovado' THEN 1 ELSE 0 END) as aprovados,
                    SUM(CASE WHEN status = 'contratado' THEN 1 ELSE 0 END) as contratados,
                    AVG(pontuacao) as pontuacao_media
                FROM curriculos
                {$whereClause}
                AND data_criacao >= DATE_SUB(CURDATE(), INTERVAL {$months} MONTH)
                GROUP BY DATE_FORMAT(data_criacao, '%Y-%m')
                ORDER BY mes ASC";
                
        $stmt = $db->prepare($sql);
        
        foreach ($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        $stmt->execute();
        $results = $stmt->fetchAll();
        
        // Preparar dados para os últimos N meses
        $monthlyData = [];
        
        for ($i = $months - 1; $i >= 0; $i--) {
            $month = date('Y-m', strtotime("-$i months"));
            $monthName = date('M/Y', strtotime("-$i months"));
            
            $monthData = [
                'mes' => $month,
                'nome' => $monthName,
                'total' => 0,
                'novos' => 0,
                'aprovados' => 0,
                'contratados' => 0,
                'pontuacao_media' => 0
            ];
            
            foreach ($results as $result) {
                if ($result['mes'] === $month) {
                    $monthData = [
                        'mes' => $month,
                        'nome' => $monthName,
                        'total' => (int)$result['total'],
                        'novos' => (int)$result['novos'],
                        'aprovados' => (int)$result['aprovados'],
                        'contratados' => (int)$result['contratados'],
                        'pontuacao_media' => round((float)$result['pontuacao_media'], 2)
                    ];
                    break;
                }
            }
            
            $monthlyData[] = $monthData;
        }
        
        return $monthlyData;
    }
    
    // Exportar dados para CSV
    public function exportCSV() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        $curriculum = new Curriculum();
        
        // Filtrar por usuário se não for admin/RH
        $filters = [];
        if (!in_array($currentUser->tipo, ['admin', 'rh'])) {
            $filters['usuario_criador_id'] = $currentUser->id;
        }
        
        // Aplicar filtros da URL
        if (!empty($_GET['status'])) {
            $filters['status'] = sanitize($_GET['status']);
        }
        
        if (!empty($_GET['grupo_id'])) {
            $filters['grupo_id'] = (int)$_GET['grupo_id'];
        }
        
        // Obter todos os currículos (sem paginação)
        $curriculums = $curriculum->getAll($filters, 1, 10000);
        
        // Gerar CSV
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=curriculos_' . date('Y-m-d') . '.csv');
        
        $output = fopen('php://output', 'w');
        
        // BOM para UTF-8
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // Cabeçalhos
        fputcsv($output, [
            'ID',
            'Nome Completo',
            'Email',
            'Telefone',
            'Cidade',
            'Estado',
            'Status',
            'Pontuação',
            'Grupo',
            'Data de Criação'
        ]);
        
        // Dados
        foreach ($curriculums as $cv) {
            fputcsv($output, [
                $cv['id'],
                $cv['nome_completo'],
                $cv['email'],
                formatPhone($cv['telefone']),
                $cv['cidade'],
                $cv['estado'],
                CURRICULUM_STATUS[$cv['status']] ?? $cv['status'],
                $cv['pontuacao'],
                $cv['grupo_nome'] ?? 'Sem grupo',
                formatDate($cv['data_criacao'], 'd/m/Y H:i')
            ]);
        }
        
        fclose($output);
        exit;
    }
}
