
<?php

require_once ROOT_PATH . '/src/models/Curriculum.php';
require_once ROOT_PATH . '/src/models/Group.php';
require_once ROOT_PATH . '/src/models/Tag.php';
require_once ROOT_PATH . '/src/services/AuthService.php';

class CurriculumController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Listar currículos
    public function index() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        $curriculum = new Curriculum();
        $group = new Group();
        $tag = new Tag();
        
        // Parâmetros de busca e filtros
        $page = (int)($_GET['page'] ?? 1);
        $search = sanitize($_GET['search'] ?? '');
        $status = sanitize($_GET['status'] ?? '');
        $grupo_id = (int)($_GET['grupo_id'] ?? 0);
        $cidade = sanitize($_GET['cidade'] ?? '');
        $sort = sanitize($_GET['sort'] ?? 'data_criacao');
        $pontuacao_min = (float)($_GET['pontuacao_min'] ?? 0);
        $pontuacao_max = (float)($_GET['pontuacao_max'] ?? 10);
        
        // Filtros
        $filters = [
            'search' => $search,
            'sort' => $sort
        ];
        
        if (!empty($status)) {
            $filters['status'] = $status;
        }
        
        if (!empty($grupo_id)) {
            $filters['grupo_id'] = $grupo_id;
        }
        
        if (!empty($cidade)) {
            $filters['cidade'] = $cidade;
        }
        
        if ($pontuacao_min > 0) {
            $filters['pontuacao_min'] = $pontuacao_min;
        }
        
        if ($pontuacao_max < 10) {
            $filters['pontuacao_max'] = $pontuacao_max;
        }
        
        // Se não for admin, filtrar por usuário
        if ($currentUser->tipo !== 'admin') {
            $filters['usuario_criador_id'] = $currentUser->id;
        }
        
        // Obter currículos
        $curriculums = $curriculum->getAll($filters, $page, ITEMS_PER_PAGE);
        $totalCurriculums = $curriculum->count($filters);
        $totalPages = ceil($totalCurriculums / ITEMS_PER_PAGE);
        
        // Obter grupos para filtro
        $groups = $group->getAll($currentUser->tipo === 'admin' ? null : $currentUser->id);
        
        // Obter cidades únicas para filtro
        $cities = $this->getUniqueCities();
        
        // Estatísticas rápidas
        $stats = $curriculum->getStats($filters);
        
        // Paginação
        $pagination = [
            'current' => $page,
            'total' => $totalPages,
            'items' => $totalCurriculums,
            'per_page' => ITEMS_PER_PAGE
        ];
        
        $title = 'Currículos';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Currículos'];
        
        require VIEWS_PATH . '/curriculos/index.php';
    }
    
    // Visualizar currículo
    public function show($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $curriculum = new Curriculum();
        
        $cv = $curriculum->findById($id);
        if (!$cv) {
            $_SESSION['error'] = 'Currículo não encontrado';
            header('Location: /curriculos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar se o usuário pode ver este currículo
        if ($currentUser->tipo !== 'admin' && $cv->usuario_criador_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para ver este currículo';
            header('Location: /curriculos');
            exit;
        }
        
        // Obter dados relacionados
        $experiencias = $cv->getExperiencias();
        $formacao = $cv->getFormacao();
        $habilidades = $cv->getHabilidades();
        $idiomas = $cv->getIdiomas();
        $arquivos = $cv->getArquivos();
        $etiquetas = $cv->getEtiquetas();
        
        $title = 'Visualizar Currículo - ' . $cv->nome_completo;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Currículos' => '/curriculos', 'Visualizar'];
        
        require VIEWS_PATH . '/curriculos/show.php';
    }
    
    // Formulário de criação
    public function create() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Obter grupos do usuário
        $group = new Group();
        $groups = $group->getAll($currentUser->id);
        
        // Obter etiquetas do usuário
        $tag = new Tag();
        $tags = $tag->getAll($currentUser->id);
        
        // Dados do formulário (em caso de erro)
        $formData = $_SESSION['form_data'] ?? [];
        unset($_SESSION['form_data']);
        
        $title = 'Novo Currículo';
        $breadcrumb = ['Dashboard' => '/dashboard', 'Currículos' => '/curriculos', 'Novo'];
        
        require VIEWS_PATH . '/curriculos/create.php';
    }
    
    // Salvar currículo
    public function store() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /curriculos/create');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /curriculos/create');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Validar e sanitizar dados
        $data = $this->validateCurriculumData($_POST);
        
        if (isset($data['errors'])) {
            $_SESSION['error'] = implode('<br>', $data['errors']);
            $_SESSION['form_data'] = $_POST;
            header('Location: /curriculos/create');
            exit;
        }
        
        // Criar currículo
        $curriculum = new Curriculum();
        
        // Mapear dados básicos
        $curriculum->nome_completo = $data['nome_completo'];
        $curriculum->email = $data['email'];
        $curriculum->telefone = $data['telefone'];
        $curriculum->celular = $data['celular'];
        $curriculum->data_nascimento = $data['data_nascimento'];
        $curriculum->endereco = $data['endereco'];
        $curriculum->cidade = $data['cidade'];
        $curriculum->estado = $data['estado'];
        $curriculum->cep = $data['cep'];
        $curriculum->objetivo = $data['objetivo'];
        $curriculum->resumo = $data['resumo'];
        $curriculum->linkedin = $data['linkedin'];
        $curriculum->github = $data['github'];
        $curriculum->website = $data['website'];
        $curriculum->usuario_criador_id = $currentUser->id;
        $curriculum->grupo_id = $data['grupo_id'];
        $curriculum->status = $data['status'] ?? 'novo';
        $curriculum->pontuacao = $data['pontuacao'] ?? 0;
        $curriculum->observacoes = $data['observacoes'];
        
        if ($curriculum->create()) {
            // Processar etiquetas
            if (!empty($data['etiquetas'])) {
                foreach ($data['etiquetas'] as $etiquetaId) {
                    $curriculum->addEtiqueta($etiquetaId);
                }
            }
            
            // Registrar no histórico
            $this->addToHistory($curriculum->id, $currentUser->id, 'Currículo criado');
            
            $_SESSION['success'] = 'Currículo criado com sucesso!';
            header("Location: /curriculos/{$curriculum->id}");
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao criar currículo';
            $_SESSION['form_data'] = $_POST;
            header('Location: /curriculos/create');
            exit;
        }
    }
    
    // Formulário de edição
    public function edit($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $curriculum = new Curriculum();
        
        $cv = $curriculum->findById($id);
        if (!$cv) {
            $_SESSION['error'] = 'Currículo não encontrado';
            header('Location: /curriculos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $cv->usuario_criador_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para editar este currículo';
            header('Location: /curriculos');
            exit;
        }
        
        // Obter grupos do usuário
        $group = new Group();
        $groups = $group->getAll($currentUser->id);
        
        // Obter etiquetas do usuário
        $tag = new Tag();
        $tags = $tag->getAll($currentUser->id);
        
        // Obter etiquetas do currículo
        $curriculumTags = $cv->getEtiquetas();
        $selectedTags = array_column($curriculumTags, 'id');
        
        // Dados do formulário (em caso de erro ou dados atuais)
        $formData = $_SESSION['form_data'] ?? [
            'nome_completo' => $cv->nome_completo,
            'email' => $cv->email,
            'telefone' => $cv->telefone,
            'celular' => $cv->celular,
            'data_nascimento' => $cv->data_nascimento,
            'endereco' => $cv->endereco,
            'cidade' => $cv->cidade,
            'estado' => $cv->estado,
            'cep' => $cv->cep,
            'objetivo' => $cv->objetivo,
            'resumo' => $cv->resumo,
            'linkedin' => $cv->linkedin,
            'github' => $cv->github,
            'website' => $cv->website,
            'grupo_id' => $cv->grupo_id,
            'status' => $cv->status,
            'pontuacao' => $cv->pontuacao,
            'observacoes' => $cv->observacoes,
            'etiquetas' => $selectedTags
        ];
        unset($_SESSION['form_data']);
        
        $title = 'Editar Currículo - ' . $cv->nome_completo;
        $breadcrumb = ['Dashboard' => '/dashboard', 'Currículos' => '/curriculos', 'Editar'];
        
        require VIEWS_PATH . '/curriculos/edit.php';
    }
    
    // Atualizar currículo
    public function update($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /curriculos');
            exit;
        }
        
        $id = (int)$params['id'];
        $curriculum = new Curriculum();
        
        $cv = $curriculum->findById($id);
        if (!$cv) {
            $_SESSION['error'] = 'Currículo não encontrado';
            header('Location: /curriculos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $cv->usuario_criador_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para editar este currículo';
            header('Location: /curriculos');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header("Location: /curriculos/{$id}/edit");
            exit;
        }
        
        // Validar dados
        $data = $this->validateCurriculumData($_POST, $id);
        
        if (isset($data['errors'])) {
            $_SESSION['error'] = implode('<br>', $data['errors']);
            $_SESSION['form_data'] = $_POST;
            header("Location: /curriculos/{$id}/edit");
            exit;
        }
        
        // Atualizar dados
        $cv->nome_completo = $data['nome_completo'];
        $cv->email = $data['email'];
        $cv->telefone = $data['telefone'];
        $cv->celular = $data['celular'];
        $cv->data_nascimento = $data['data_nascimento'];
        $cv->endereco = $data['endereco'];
        $cv->cidade = $data['cidade'];
        $cv->estado = $data['estado'];
        $cv->cep = $data['cep'];
        $cv->objetivo = $data['objetivo'];
        $cv->resumo = $data['resumo'];
        $cv->linkedin = $data['linkedin'];
        $cv->github = $data['github'];
        $cv->website = $data['website'];
        $cv->grupo_id = $data['grupo_id'];
        $cv->status = $data['status'] ?? $cv->status;
        $cv->pontuacao = $data['pontuacao'] ?? $cv->pontuacao;
        $cv->observacoes = $data['observacoes'];
        
        if ($cv->update()) {
            // Atualizar etiquetas
            // Remover todas as etiquetas atuais
            $this->removeAllTags($cv->id);
            
            // Adicionar novas etiquetas
            if (!empty($data['etiquetas'])) {
                foreach ($data['etiquetas'] as $etiquetaId) {
                    $cv->addEtiqueta($etiquetaId);
                }
            }
            
            // Registrar no histórico
            $this->addToHistory($cv->id, $currentUser->id, 'Currículo atualizado');
            
            $_SESSION['success'] = 'Currículo atualizado com sucesso!';
            header("Location: /curriculos/{$cv->id}");
            exit;
        } else {
            $_SESSION['error'] = 'Erro ao atualizar currículo';
            $_SESSION['form_data'] = $_POST;
            header("Location: /curriculos/{$id}/edit");
            exit;
        }
    }
    
    // Excluir currículo
    public function delete($params) {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        $id = (int)$params['id'];
        $curriculum = new Curriculum();
        
        $cv = $curriculum->findById($id);
        if (!$cv) {
            $_SESSION['error'] = 'Currículo não encontrado';
            header('Location: /curriculos');
            exit;
        }
        
        $currentUser = $this->authService->getCurrentUser();
        
        // Verificar permissão
        if ($currentUser->tipo !== 'admin' && $cv->usuario_criador_id != $currentUser->id) {
            $_SESSION['error'] = 'Você não tem permissão para excluir este currículo';
            header('Location: /curriculos');
            exit;
        }
        
        if ($cv->delete()) {
            $_SESSION['success'] = 'Currículo excluído com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao excluir currículo';
        }
        
        header('Location: /curriculos');
        exit;
    }
    
    // Validar dados do currículo
    private function validateCurriculumData($data, $excludeId = null) {
        $errors = [];
        
        // Validações obrigatórias
        if (empty($data['nome_completo'])) {
            $errors[] = 'Nome completo é obrigatório';
        }
        
        if (empty($data['email']) || !isValidEmail($data['email'])) {
            $errors[] = 'Email válido é obrigatório';
        }
        
        // Verificar se email já existe
        if (!empty($data['email']) && $this->emailExists($data['email'], $excludeId)) {
            $errors[] = 'Este email já está cadastrado';
        }
        
        // Validar data de nascimento
        if (!empty($data['data_nascimento'])) {
            $birthDate = DateTime::createFromFormat('Y-m-d', $data['data_nascimento']);
            if (!$birthDate || $birthDate > new DateTime()) {
                $errors[] = 'Data de nascimento inválida';
            }
        }
        
        // Validar pontuação
        if (!empty($data['pontuacao'])) {
            $pontuacao = (float)$data['pontuacao'];
            if ($pontuacao < 0 || $pontuacao > 10) {
                $errors[] = 'Pontuação deve estar entre 0 e 10';
            }
        }
        
        if (!empty($errors)) {
            return ['errors' => $errors];
        }
        
        // Sanitizar dados
        return [
            'nome_completo' => sanitize($data['nome_completo']),
            'email' => sanitize($data['email']),
            'telefone' => sanitize($data['telefone'] ?? ''),
            'celular' => sanitize($data['celular'] ?? ''),
            'data_nascimento' => !empty($data['data_nascimento']) ? $data['data_nascimento'] : null,
            'endereco' => sanitize($data['endereco'] ?? ''),
            'cidade' => sanitize($data['cidade'] ?? ''),
            'estado' => sanitize($data['estado'] ?? ''),
            'cep' => sanitize($data['cep'] ?? ''),
            'objetivo' => sanitize($data['objetivo'] ?? ''),
            'resumo' => sanitize($data['resumo'] ?? ''),
            'linkedin' => sanitize($data['linkedin'] ?? ''),
            'github' => sanitize($data['github'] ?? ''),
            'website' => sanitize($data['website'] ?? ''),
            'grupo_id' => !empty($data['grupo_id']) ? (int)$data['grupo_id'] : null,
            'status' => sanitize($data['status'] ?? 'novo'),
            'pontuacao' => !empty($data['pontuacao']) ? (float)$data['pontuacao'] : 0,
            'observacoes' => sanitize($data['observacoes'] ?? ''),
            'etiquetas' => $data['etiquetas'] ?? []
        ];
    }
    
    // Verificar se email já existe
    private function emailExists($email, $excludeId = null) {
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT id FROM curriculos WHERE email = :email";
        $params = [':email' => $email];
        
        if ($excludeId) {
            $sql .= " AND id != :exclude_id";
            $params[':exclude_id'] = $excludeId;
        }
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetch() !== false;
    }
    
    // Obter cidades únicas
    private function getUniqueCities() {
        $db = Database::getInstance()->getConnection();
        
        $sql = "SELECT DISTINCT cidade FROM curriculos 
                WHERE cidade IS NOT NULL AND cidade != '' 
                ORDER BY cidade ASC";
        $stmt = $db->query($sql);
        
        return array_column($stmt->fetchAll(), 'cidade');
    }
    
    // Remover todas as etiquetas do currículo
    private function removeAllTags($curriculumId) {
        $db = Database::getInstance()->getConnection();
        
        $sql = "DELETE FROM curriculo_etiquetas WHERE curriculo_id = :curriculo_id";
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $curriculumId);
        
        return $stmt->execute();
    }
    
    // Adicionar ao histórico
    private function addToHistory($curriculumId, $userId, $action, $details = null) {
        $db = Database::getInstance()->getConnection();
        
        $sql = "INSERT INTO historico_alteracoes (curriculo_id, usuario_id, acao, detalhes) 
                VALUES (:curriculo_id, :usuario_id, :acao, :detalhes)";
                
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':curriculo_id', $curriculumId);
        $stmt->bindParam(':usuario_id', $userId);
        $stmt->bindParam(':acao', $action);
        $stmt->bindParam(':detalhes', json_encode($details));
        
        return $stmt->execute();
    }
}
