
<?php

require_once ROOT_PATH . '/src/services/AuthService.php';

class AuthController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Exibir formulário de login
    public function showLogin() {
        // Se já está logado, redirecionar para dashboard
        if ($this->authService->isLoggedIn()) {
            header('Location: /dashboard');
            exit;
        }
        
        $title = 'Login';
        require VIEWS_PATH . '/auth/login.php';
    }
    
    // Processar login
    public function processLogin() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /login');
            exit;
        }
        
        $email = sanitize($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $remember = isset($_POST['remember']);
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /login');
            exit;
        }
        
        if (empty($email) || empty($password)) {
            $_SESSION['error'] = 'Email e senha são obrigatórios';
            header('Location: /login');
            exit;
        }
        
        $result = $this->authService->login($email, $password, $remember);
        
        if ($result['success']) {
            $_SESSION['success'] = 'Login realizado com sucesso!';
            
            // Redirecionar para página desejada ou dashboard
            $redirect = $_SESSION['redirect_after_login'] ?? '/dashboard';
            unset($_SESSION['redirect_after_login']);
            
            header("Location: $redirect");
            exit;
        } else {
            $_SESSION['error'] = $result['message'];
            header('Location: /login');
            exit;
        }
    }
    
    // Exibir formulário de registro
    public function showRegister() {
        // Se já está logado, redirecionar para dashboard
        if ($this->authService->isLoggedIn()) {
            header('Location: /dashboard');
            exit;
        }
        
        $title = 'Registro';
        require VIEWS_PATH . '/auth/register.php';
    }
    
    // Processar registro
    public function processRegister() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /register');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /register');
            exit;
        }
        
        $userData = [
            'nome' => sanitize($_POST['nome'] ?? ''),
            'email' => sanitize($_POST['email'] ?? ''),
            'senha' => $_POST['senha'] ?? '',
            'senha_confirmacao' => $_POST['senha_confirmacao'] ?? '',
            'tipo' => sanitize($_POST['tipo'] ?? 'recrutador'),
            'telefone' => sanitize($_POST['telefone'] ?? ''),
            'departamento' => sanitize($_POST['departamento'] ?? '')
        ];
        
        // Validações
        $errors = [];
        
        if (empty($userData['nome'])) {
            $errors[] = 'Nome é obrigatório';
        }
        
        if (empty($userData['email']) || !isValidEmail($userData['email'])) {
            $errors[] = 'Email válido é obrigatório';
        }
        
        if (empty($userData['senha']) || strlen($userData['senha']) < PASSWORD_MIN_LENGTH) {
            $errors[] = 'Senha deve ter pelo menos ' . PASSWORD_MIN_LENGTH . ' caracteres';
        }
        
        if ($userData['senha'] !== $userData['senha_confirmacao']) {
            $errors[] = 'Confirmação de senha não confere';
        }
        
        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            $_SESSION['form_data'] = $userData;
            header('Location: /register');
            exit;
        }
        
        $result = $this->authService->register($userData);
        
        if ($result['success']) {
            $_SESSION['success'] = 'Registro realizado com sucesso! Faça login para continuar.';
            header('Location: /login');
            exit;
        } else {
            $_SESSION['error'] = $result['message'];
            $_SESSION['form_data'] = $userData;
            header('Location: /register');
            exit;
        }
    }
    
    // Fazer logout
    public function logout() {
        $this->authService->logout();
        $_SESSION['success'] = 'Logout realizado com sucesso!';
        header('Location: /login');
        exit;
    }
    
    // Exibir perfil do usuário
    public function profile() {
        if (!$this->authService->isLoggedIn()) {
            $_SESSION['redirect_after_login'] = '/profile';
            header('Location: /login');
            exit;
        }
        
        $user = $this->authService->getCurrentUser();
        $title = 'Meu Perfil';
        
        require VIEWS_PATH . '/auth/profile.php';
    }
    
    // Atualizar perfil
    public function updateProfile() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /profile');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /profile');
            exit;
        }
        
        $user = $this->authService->getCurrentUser();
        
        $userData = [
            'nome' => sanitize($_POST['nome'] ?? ''),
            'email' => sanitize($_POST['email'] ?? ''),
            'telefone' => sanitize($_POST['telefone'] ?? ''),
            'departamento' => sanitize($_POST['departamento'] ?? '')
        ];
        
        // Validações
        $errors = [];
        
        if (empty($userData['nome'])) {
            $errors[] = 'Nome é obrigatório';
        }
        
        if (empty($userData['email']) || !isValidEmail($userData['email'])) {
            $errors[] = 'Email válido é obrigatório';
        }
        
        // Verificar se email já existe (para outro usuário)
        $userModel = new User();
        if ($userModel->emailExists($userData['email'], $user->id)) {
            $errors[] = 'Este email já está em uso';
        }
        
        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            header('Location: /profile');
            exit;
        }
        
        // Atualizar dados
        $user->nome = $userData['nome'];
        $user->email = $userData['email'];
        $user->telefone = $userData['telefone'];
        $user->departamento = $userData['departamento'];
        
        if ($user->update()) {
            // Atualizar dados da sessão
            $_SESSION['user_name'] = $user->nome;
            $_SESSION['user_email'] = $user->email;
            
            $_SESSION['success'] = 'Perfil atualizado com sucesso!';
        } else {
            $_SESSION['error'] = 'Erro ao atualizar perfil';
        }
        
        header('Location: /profile');
        exit;
    }
    
    // Alterar senha
    public function changePassword() {
        if (!$this->authService->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /profile');
            exit;
        }
        
        // Validar CSRF
        if (!verifyCsrfToken($_POST['csrf_token'] ?? '')) {
            $_SESSION['error'] = 'Token de segurança inválido';
            header('Location: /profile');
            exit;
        }
        
        $currentPassword = $_POST['senha_atual'] ?? '';
        $newPassword = $_POST['nova_senha'] ?? '';
        $confirmPassword = $_POST['confirmar_senha'] ?? '';
        
        // Validações
        $errors = [];
        
        if (empty($currentPassword)) {
            $errors[] = 'Senha atual é obrigatória';
        }
        
        if (empty($newPassword) || strlen($newPassword) < PASSWORD_MIN_LENGTH) {
            $errors[] = 'Nova senha deve ter pelo menos ' . PASSWORD_MIN_LENGTH . ' caracteres';
        }
        
        if ($newPassword !== $confirmPassword) {
            $errors[] = 'Confirmação de senha não confere';
        }
        
        if (!empty($errors)) {
            $_SESSION['error'] = implode('<br>', $errors);
            header('Location: /profile');
            exit;
        }
        
        $result = $this->authService->changePassword($currentPassword, $newPassword);
        
        if ($result['success']) {
            $_SESSION['success'] = $result['message'];
        } else {
            $_SESSION['error'] = $result['message'];
        }
        
        header('Location: /profile');
        exit;
    }
    
    // Verificar se está autenticado (middleware)
    public function requireAuth() {
        if (!$this->authService->isLoggedIn()) {
            $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
            header('Location: /login');
            exit;
        }
    }
    
    // Verificar permissão (middleware)
    public function requirePermission($permission) {
        $this->requireAuth();
        
        if (!$this->authService->hasPermission($permission)) {
            http_response_code(403);
            $_SESSION['error'] = 'Você não tem permissão para acessar esta página';
            header('Location: /dashboard');
            exit;
        }
    }
}
