
<?php

require_once ROOT_PATH . '/src/services/AuthService.php';

class ApiController {
    private $authService;
    
    public function __construct() {
        $this->authService = new AuthService();
    }
    
    // Buscar CEP via API
    public function getCep($params) {
        if (!$this->authService->isLoggedIn()) {
            http_response_code(401);
            echo json_encode(['error' => 'Não autorizado']);
            exit;
        }
        
        $cep = preg_replace('/\D/', '', $params['cep']);
        
        if (strlen($cep) !== 8) {
            http_response_code(400);
            echo json_encode(['error' => 'CEP inválido']);
            exit;
        }
        
        // Buscar via ViaCEP
        $url = "https://viacep.com.br/ws/{$cep}/json/";
        $response = file_get_contents($url);
        
        if ($response === false) {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao consultar CEP']);
            exit;
        }
        
        $data = json_decode($response, true);
        
        if (isset($data['erro'])) {
            http_response_code(404);
            echo json_encode(['error' => 'CEP não encontrado']);
            exit;
        }
        
        echo json_encode([
            'success' => true,
            'data' => [
                'cep' => $data['cep'],
                'endereco' => $data['logradouro'],
                'bairro' => $data['bairro'],
                'cidade' => $data['localidade'],
                'estado' => $data['uf']
            ]
        ]);
    }
    
    // Busca dinâmica
    public function search() {
        if (!$this->authService->isLoggedIn()) {
            http_response_code(401);
            echo json_encode(['error' => 'Não autorizado']);
            exit;
        }
        
        $query = sanitize($_GET['q'] ?? '');
        $type = sanitize($_GET['type'] ?? 'curriculos');
        $limit = min((int)($_GET['limit'] ?? 10), 50);
        
        if (strlen($query) < 2) {
            echo json_encode(['results' => []]);
            exit;
        }
        
        $db = Database::getInstance()->getConnection();
        $currentUser = $this->authService->getCurrentUser();
        
        $results = [];
        
        switch ($type) {
            case 'curriculos':
                $sql = "SELECT id, nome_completo as label, email, telefone 
                        FROM curriculos 
                        WHERE (nome_completo LIKE :query OR email LIKE :query OR telefone LIKE :query)";
                
                // Filtrar por usuário se não for admin
                if ($currentUser->tipo !== 'admin') {
                    $sql .= " AND usuario_criador_id = :user_id";
                }
                
                $sql .= " ORDER BY nome_completo LIMIT :limit";
                
                $stmt = $db->prepare($sql);
                $stmt->bindValue(':query', "%{$query}%");
                $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
                
                if ($currentUser->tipo !== 'admin') {
                    $stmt->bindValue(':user_id', $currentUser->id);
                }
                
                $stmt->execute();
                $results = $stmt->fetchAll();
                
                // Adicionar URL para cada resultado
                foreach ($results as &$result) {
                    $result['url'] = "/curriculos/{$result['id']}";
                    $result['type'] = 'currículo';
                }
                break;
                
            case 'grupos':
                $sql = "SELECT id, nome as label, descricao, cor 
                        FROM grupos 
                        WHERE nome LIKE :query";
                
                // Filtrar por usuário se não for admin
                if ($currentUser->tipo !== 'admin') {
                    $sql .= " AND usuario_id = :user_id";
                }
                
                $sql .= " ORDER BY nome LIMIT :limit";
                
                $stmt = $db->prepare($sql);
                $stmt->bindValue(':query', "%{$query}%");
                $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
                
                if ($currentUser->tipo !== 'admin') {
                    $stmt->bindValue(':user_id', $currentUser->id);
                }
                
                $stmt->execute();
                $results = $stmt->fetchAll();
                
                // Adicionar URL para cada resultado
                foreach ($results as &$result) {
                    $result['url'] = "/grupos/{$result['id']}";
                    $result['type'] = 'grupo';
                }
                break;
                
            case 'etiquetas':
                $sql = "SELECT id, nome as label, cor 
                        FROM etiquetas 
                        WHERE nome LIKE :query";
                
                // Filtrar por usuário se não for admin
                if ($currentUser->tipo !== 'admin') {
                    $sql .= " AND usuario_id = :user_id";
                }
                
                $sql .= " ORDER BY nome LIMIT :limit";
                
                $stmt = $db->prepare($sql);
                $stmt->bindValue(':query', "%{$query}%");
                $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
                
                if ($currentUser->tipo !== 'admin') {
                    $stmt->bindValue(':user_id', $currentUser->id);
                }
                
                $stmt->execute();
                $results = $stmt->fetchAll();
                
                // Adicionar URL para cada resultado
                foreach ($results as &$result) {
                    $result['url'] = "/etiquetas/{$result['id']}/edit";
                    $result['type'] = 'etiqueta';
                }
                break;
                
            case 'usuarios':
                // Apenas admin e RH podem buscar usuários
                if (!in_array($currentUser->tipo, ['admin', 'rh'])) {
                    http_response_code(403);
                    echo json_encode(['error' => 'Sem permissão']);
                    exit;
                }
                
                $sql = "SELECT id, nome as label, email, tipo 
                        FROM usuarios 
                        WHERE (nome LIKE :query OR email LIKE :query) AND ativo = 1
                        ORDER BY nome LIMIT :limit";
                
                $stmt = $db->prepare($sql);
                $stmt->bindValue(':query', "%{$query}%");
                $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
                $stmt->execute();
                $results = $stmt->fetchAll();
                
                // Adicionar URL para cada resultado
                foreach ($results as &$result) {
                    $result['url'] = "/usuarios/{$result['id']}";
                    $result['type'] = 'usuário';
                }
                break;
        }
        
        echo json_encode(['results' => $results]);
    }
}
