
-- ============================================
-- Script de Inicialização para RcTecNet
-- Database: rivadavi_ger_curriculos
-- Host: 209.126.87.237
-- ============================================

-- Usar o banco de dados específico da RcTecNet
USE rivadavi_ger_curriculos;

-- Tabela de usuários
CREATE TABLE IF NOT EXISTS usuarios (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome VARCHAR(255) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    senha VARCHAR(255) NOT NULL,
    tipo ENUM('admin', 'rh', 'recrutador') DEFAULT 'recrutador',
    ativo BOOLEAN DEFAULT TRUE,
    foto VARCHAR(255) NULL,
    telefone VARCHAR(20) NULL,
    departamento VARCHAR(100) NULL,
    data_criacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    data_atualizacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tabela de grupos
CREATE TABLE IF NOT EXISTS grupos (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome VARCHAR(255) NOT NULL,
    descricao TEXT NULL,
    cor VARCHAR(7) DEFAULT '#3498db',
    usuario_id INT NOT NULL,
    data_criacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE
);

-- Tabela de etiquetas
CREATE TABLE IF NOT EXISTS etiquetas (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome VARCHAR(100) NOT NULL,
    cor VARCHAR(7) DEFAULT '#95a5a6',
    usuario_id INT NOT NULL,
    data_criacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    UNIQUE KEY unique_etiqueta_usuario (nome, usuario_id)
);

-- Tabela principal de currículos
CREATE TABLE IF NOT EXISTS curriculos (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome_completo VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    telefone VARCHAR(20) NULL,
    celular VARCHAR(20) NULL,
    data_nascimento DATE NULL,
    cpf VARCHAR(14) NULL,
    estado_civil ENUM('solteiro', 'casado', 'divorciado', 'viuvo') NULL,
    endereco TEXT NULL,
    cidade VARCHAR(100) NULL,
    estado VARCHAR(2) NULL,
    cep VARCHAR(10) NULL,
    objetivo TEXT NULL,
    resumo_profissional TEXT NULL,
    pretensao_salarial DECIMAL(10,2) NULL,
    disponibilidade ENUM('imediata', '15_dias', '30_dias', 'a_combinar') DEFAULT 'a_combinar',
    status ENUM('novo', 'triagem', 'aprovado', 'rejeitado', 'entrevista', 'contratado') DEFAULT 'novo',
    pontuacao INT DEFAULT 0,
    arquivo_curriculo VARCHAR(255) NULL,
    foto VARCHAR(255) NULL,
    observacoes TEXT NULL,
    usuario_id INT NOT NULL,
    grupo_id INT NULL,
    data_criacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    data_atualizacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    FOREIGN KEY (grupo_id) REFERENCES grupos(id) ON DELETE SET NULL,
    INDEX idx_email (email),
    INDEX idx_status (status),
    INDEX idx_usuario (usuario_id),
    INDEX idx_grupo (grupo_id)
);

-- Tabela de experiências profissionais
CREATE TABLE IF NOT EXISTS experiencias (
    id INT PRIMARY KEY AUTO_INCREMENT,
    curriculo_id INT NOT NULL,
    empresa VARCHAR(255) NOT NULL,
    cargo VARCHAR(255) NOT NULL,
    data_inicio DATE NOT NULL,
    data_fim DATE NULL,
    atual BOOLEAN DEFAULT FALSE,
    descricao TEXT NULL,
    salario DECIMAL(10,2) NULL,
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    INDEX idx_curriculo (curriculo_id)
);

-- Tabela de formação acadêmica
CREATE TABLE IF NOT EXISTS formacao (
    id INT PRIMARY KEY AUTO_INCREMENT,
    curriculo_id INT NOT NULL,
    instituicao VARCHAR(255) NOT NULL,
    curso VARCHAR(255) NOT NULL,
    nivel ENUM('fundamental', 'medio', 'tecnico', 'superior', 'pos', 'mestrado', 'doutorado') NOT NULL,
    situacao ENUM('completo', 'cursando', 'incompleto') NOT NULL,
    data_inicio DATE NULL,
    data_conclusao DATE NULL,
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    INDEX idx_curriculo (curriculo_id)
);

-- Tabela de cursos complementares
CREATE TABLE IF NOT EXISTS cursos (
    id INT PRIMARY KEY AUTO_INCREMENT,
    curriculo_id INT NOT NULL,
    nome VARCHAR(255) NOT NULL,
    instituicao VARCHAR(255) NULL,
    carga_horaria INT NULL,
    data_conclusao DATE NULL,
    certificado VARCHAR(255) NULL,
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    INDEX idx_curriculo (curriculo_id)
);

-- Tabela de habilidades
CREATE TABLE IF NOT EXISTS habilidades (
    id INT PRIMARY KEY AUTO_INCREMENT,
    curriculo_id INT NOT NULL,
    nome VARCHAR(255) NOT NULL,
    nivel ENUM('basico', 'intermediario', 'avancado', 'expert') DEFAULT 'intermediario',
    tipo ENUM('tecnica', 'comportamental', 'idioma') DEFAULT 'tecnica',
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    INDEX idx_curriculo (curriculo_id),
    INDEX idx_tipo (tipo)
);

-- Tabela de idiomas
CREATE TABLE IF NOT EXISTS idiomas (
    id INT PRIMARY KEY AUTO_INCREMENT,
    curriculo_id INT NOT NULL,
    idioma VARCHAR(100) NOT NULL,
    nivel_conversacao ENUM('basico', 'intermediario', 'avancado', 'fluente', 'nativo') DEFAULT 'basico',
    nivel_leitura ENUM('basico', 'intermediario', 'avancado', 'fluente', 'nativo') DEFAULT 'basico',
    nivel_escrita ENUM('basico', 'intermediario', 'avancado', 'fluente', 'nativo') DEFAULT 'basico',
    certificado VARCHAR(255) NULL,
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    INDEX idx_curriculo (curriculo_id)
);

-- Tabela de relacionamento curriculos-etiquetas (many-to-many)
CREATE TABLE IF NOT EXISTS curriculo_etiquetas (
    curriculo_id INT NOT NULL,
    etiqueta_id INT NOT NULL,
    PRIMARY KEY (curriculo_id, etiqueta_id),
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    FOREIGN KEY (etiqueta_id) REFERENCES etiquetas(id) ON DELETE CASCADE
);

-- Tabela de histórico de status
CREATE TABLE IF NOT EXISTS historico_status (
    id INT PRIMARY KEY AUTO_INCREMENT,
    curriculo_id INT NOT NULL,
    status_anterior ENUM('novo', 'triagem', 'aprovado', 'rejeitado', 'entrevista', 'contratado'),
    status_novo ENUM('novo', 'triagem', 'aprovado', 'rejeitado', 'entrevista', 'contratado') NOT NULL,
    observacoes TEXT NULL,
    usuario_id INT NOT NULL,
    data_alteracao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (curriculo_id) REFERENCES curriculos(id) ON DELETE CASCADE,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    INDEX idx_curriculo (curriculo_id),
    INDEX idx_data (data_alteracao)
);

-- Tabela de logs do sistema
CREATE TABLE IF NOT EXISTS logs_sistema (
    id INT PRIMARY KEY AUTO_INCREMENT,
    usuario_id INT NULL,
    acao VARCHAR(255) NOT NULL,
    tabela VARCHAR(100) NULL,
    registro_id INT NULL,
    dados_anteriores JSON NULL,
    dados_novos JSON NULL,
    ip VARCHAR(45) NULL,
    user_agent TEXT NULL,
    data_acao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_usuario (usuario_id),
    INDEX idx_data (data_acao),
    INDEX idx_tabela (tabela)
);

-- Inserir dados iniciais para teste (opcional)
-- Usuário administrador padrão (senha: admin123)
INSERT IGNORE INTO usuarios (nome, email, senha, tipo) VALUES 
('Administrador', 'admin@rivadavia.com.br', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin');

-- Grupos padrão
INSERT IGNORE INTO grupos (nome, descricao, cor, usuario_id) VALUES 
('Administrativo', 'Cargos administrativos e de escritório', '#3498db', 1),
('Técnico', 'Cargos técnicos e especializados', '#e74c3c', 1),
('Comercial', 'Área comercial e vendas', '#f39c12', 1),
('Operacional', 'Cargos operacionais', '#27ae60', 1);

-- Etiquetas padrão
INSERT IGNORE INTO etiquetas (nome, cor, usuario_id) VALUES 
('Urgente', '#e74c3c', 1),
('Experiência', '#3498db', 1),
('Formação Superior', '#9b59b6', 1),
('Idiomas', '#f39c12', 1),
('Liderança', '#27ae60', 1);

-- Configurações do banco específicas para RcTecNet
-- Otimizações de performance
SET GLOBAL innodb_buffer_pool_size = 128M;
SET GLOBAL max_connections = 200;
SET GLOBAL wait_timeout = 600;
SET GLOBAL interactive_timeout = 600;

-- Configurações de charset para garantir compatibilidade
ALTER DATABASE rivadavi_ger_curriculos CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
