
// Custom JavaScript for Gerenciador de Currículos

document.addEventListener('DOMContentLoaded', function() {
    // Initialize components
    initSearch();
    initFileUpload();
    initFormValidation();
    initTooltips();
    initMasks();
    initCharts();
    
    // Auto-hide alerts
    setTimeout(function() {
        const alerts = document.querySelectorAll('.alert');
        alerts.forEach(alert => {
            alert.style.transition = 'opacity 0.5s ease';
            alert.style.opacity = '0';
            setTimeout(() => alert.remove(), 500);
        });
    }, 5000);
});

// Search functionality
function initSearch() {
    const searchInputs = document.querySelectorAll('.search-input');
    
    searchInputs.forEach(input => {
        let timeout;
        
        input.addEventListener('input', function() {
            clearTimeout(timeout);
            const query = this.value.trim();
            
            if (query.length < 2) {
                hideSearchResults();
                return;
            }
            
            timeout = setTimeout(() => {
                performSearch(query, this);
            }, 300);
        });
        
        // Hide results when clicking outside
        document.addEventListener('click', function(e) {
            if (!input.contains(e.target)) {
                hideSearchResults();
            }
        });
    });
}

function performSearch(query, input) {
    const type = input.dataset.searchType || 'curriculos';
    const url = `/api/search?q=${encodeURIComponent(query)}&type=${type}&limit=10`;
    
    fetch(url)
        .then(response => response.json())
        .then(data => {
            showSearchResults(data.results || [], input);
        })
        .catch(error => {
            console.error('Search error:', error);
        });
}

function showSearchResults(results, input) {
    const container = input.parentElement;
    let resultsDiv = container.querySelector('.search-results');
    
    if (!resultsDiv) {
        resultsDiv = document.createElement('div');
        resultsDiv.className = 'search-results';
        container.appendChild(resultsDiv);
    }
    
    if (results.length === 0) {
        resultsDiv.innerHTML = '<div class="search-result-item text-muted">Nenhum resultado encontrado</div>';
    } else {
        resultsDiv.innerHTML = results.map(item => `
            <div class="search-result-item" onclick="window.location.href='${item.url}'">
                <div class="d-flex justify-content-between">
                    <span>${item.label}</span>
                    <small class="text-muted">${item.type}</small>
                </div>
                ${item.email ? `<small class="text-muted">${item.email}</small>` : ''}
            </div>
        `).join('');
    }
    
    resultsDiv.style.display = 'block';
}

function hideSearchResults() {
    const resultsContainers = document.querySelectorAll('.search-results');
    resultsContainers.forEach(container => {
        container.style.display = 'none';
    });
}

// File upload functionality
function initFileUpload() {
    const uploadAreas = document.querySelectorAll('.file-upload-area');
    
    uploadAreas.forEach(area => {
        const input = area.querySelector('input[type="file"]');
        
        if (input) {
            area.addEventListener('click', () => input.click());
            
            area.addEventListener('dragover', function(e) {
                e.preventDefault();
                this.classList.add('dragover');
            });
            
            area.addEventListener('dragleave', function() {
                this.classList.remove('dragover');
            });
            
            area.addEventListener('drop', function(e) {
                e.preventDefault();
                this.classList.remove('dragover');
                
                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    input.files = files;
                    handleFileUpload(input);
                }
            });
            
            input.addEventListener('change', function() {
                handleFileUpload(this);
            });
        }
    });
}

function handleFileUpload(input) {
    const files = Array.from(input.files);
    const maxSize = 5 * 1024 * 1024; // 5MB
    const allowedTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'image/jpeg', 'image/png'];
    
    const validFiles = [];
    const errors = [];
    
    files.forEach(file => {
        if (file.size > maxSize) {
            errors.push(`${file.name}: Arquivo muito grande (máximo 5MB)`);
            return;
        }
        
        if (!allowedTypes.includes(file.type)) {
            errors.push(`${file.name}: Tipo de arquivo não permitido`);
            return;
        }
        
        validFiles.push(file);
    });
    
    if (errors.length > 0) {
        showAlert('danger', errors.join('<br>'));
        return;
    }
    
    if (validFiles.length > 0) {
        uploadFiles(validFiles, input);
    }
}

function uploadFiles(files, input) {
    const formData = new FormData();
    const curriculumId = input.dataset.curriculumId;
    const tipo = input.dataset.tipo || 'curriculo';
    
    files.forEach(file => {
        formData.append('arquivo', file);
    });
    
    formData.append('curriculum_id', curriculumId);
    formData.append('tipo', tipo);
    
    showLoading(true);
    
    fetch('/upload', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('success', data.message);
            // Refresh file list if exists
            const fileList = document.querySelector('.file-list');
            if (fileList) {
                location.reload();
            }
        } else {
            showAlert('danger', data.error);
        }
    })
    .catch(error => {
        console.error('Upload error:', error);
        showAlert('danger', 'Erro ao enviar arquivo');
    })
    .finally(() => {
        showLoading(false);
    });
}

// Form validation
function initFormValidation() {
    const forms = document.querySelectorAll('form[data-validate]');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
                return false;
            }
            
            const submitBtn = this.querySelector('button[type="submit"]');
            if (submitBtn) {
                showButtonLoading(submitBtn);
            }
        });
        
        // Real-time validation
        const inputs = form.querySelectorAll('input, select, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                validateField(this);
            });
        });
    });
}

function validateForm(form) {
    const inputs = form.querySelectorAll('input[required], select[required], textarea[required]');
    let isValid = true;
    
    inputs.forEach(input => {
        if (!validateField(input)) {
            isValid = false;
        }
    });
    
    return isValid;
}

function validateField(field) {
    const value = field.value.trim();
    const type = field.type;
    let isValid = true;
    let message = '';
    
    // Required check
    if (field.hasAttribute('required') && !value) {
        isValid = false;
        message = 'Este campo é obrigatório';
    }
    
    // Type-specific validation
    if (value && !isValid === false) {
        switch (type) {
            case 'email':
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailRegex.test(value)) {
                    isValid = false;
                    message = 'Email inválido';
                }
                break;
                
            case 'tel':
                const phoneRegex = /^\(\d{2}\)\s\d{4,5}-\d{4}$/;
                if (value && !phoneRegex.test(value)) {
                    isValid = false;
                    message = 'Telefone inválido';
                }
                break;
                
            case 'password':
                if (field.name === 'senha' && value.length < 6) {
                    isValid = false;
                    message = 'Senha deve ter pelo menos 6 caracteres';
                }
                
                // Password confirmation
                if (field.name === 'senha_confirmacao') {
                    const senhaField = document.querySelector('input[name="senha"]');
                    if (senhaField && value !== senhaField.value) {
                        isValid = false;
                        message = 'As senhas não conferem';
                    }
                }
                break;
        }
    }
    
    // Custom validation
    const customValidation = field.dataset.validation;
    if (customValidation) {
        try {
            const validationFn = new Function('value', customValidation);
            const customResult = validationFn(value);
            if (customResult !== true) {
                isValid = false;
                message = customResult || 'Valor inválido';
            }
        } catch (e) {
            console.error('Custom validation error:', e);
        }
    }
    
    // Update field appearance
    field.classList.toggle('is-invalid', !isValid);
    field.classList.toggle('is-valid', isValid && value);
    
    // Show/hide error message
    let feedback = field.parentElement.querySelector('.invalid-feedback');
    if (!isValid && message) {
        if (!feedback) {
            feedback = document.createElement('div');
            feedback.className = 'invalid-feedback';
            field.parentElement.appendChild(feedback);
        }
        feedback.textContent = message;
    } else if (feedback) {
        feedback.remove();
    }
    
    return isValid;
}

// Input masks
function initMasks() {
    // Phone mask
    const phoneInputs = document.querySelectorAll('input[type="tel"], input[name*="telefone"], input[name*="celular"]');
    phoneInputs.forEach(input => {
        input.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            
            if (value.length >= 11) {
                value = value.replace(/(\d{2})(\d{5})(\d{4})/, '($1) $2-$3');
            } else if (value.length >= 7) {
                value = value.replace(/(\d{2})(\d{4})(\d{0,4})/, '($1) $2-$3');
            } else if (value.length >= 3) {
                value = value.replace(/(\d{2})(\d{0,5})/, '($1) $2');
            }
            
            e.target.value = value;
        });
    });
    
    // CEP mask
    const cepInputs = document.querySelectorAll('input[name="cep"]');
    cepInputs.forEach(input => {
        input.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            
            if (value.length >= 5) {
                value = value.replace(/(\d{5})(\d{0,3})/, '$1-$2');
            }
            
            e.target.value = value;
        });
        
        // CEP lookup
        input.addEventListener('blur', function() {
            const cep = this.value.replace(/\D/g, '');
            if (cep.length === 8) {
                lookupCep(cep);
            }
        });
    });
}

function lookupCep(cep) {
    fetch(`/api/cep/${cep}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const fields = data.data;
                
                Object.keys(fields).forEach(key => {
                    const input = document.querySelector(`input[name="${key}"]`);
                    if (input && !input.value) {
                        input.value = fields[key];
                    }
                });
            }
        })
        .catch(error => {
            console.error('CEP lookup error:', error);
        });
}

// Tooltips
function initTooltips() {
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function(tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
}

// Charts initialization
function initCharts() {
    // Set global Chart.js defaults
    if (typeof Chart !== 'undefined') {
        Chart.defaults.font.family = "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif";
        Chart.defaults.color = '#6c757d';
        Chart.defaults.plugins.legend.labels.usePointStyle = true;
    }
}

// Utility functions
function showAlert(type, message) {
    const alertsContainer = document.getElementById('alerts-container') || document.body;
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type} alert-dismissible fade show`;
    alert.innerHTML = `
        <i class="bi bi-${getAlertIcon(type)} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    alertsContainer.appendChild(alert);
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

function getAlertIcon(type) {
    const icons = {
        'success': 'check-circle',
        'danger': 'exclamation-triangle',
        'warning': 'exclamation-triangle',
        'info': 'info-circle'
    };
    
    return icons[type] || 'info-circle';
}

function showLoading(show) {
    let overlay = document.getElementById('loading-overlay');
    
    if (show) {
        if (!overlay) {
            overlay = document.createElement('div');
            overlay.id = 'loading-overlay';
            overlay.className = 'loading-overlay';
            overlay.innerHTML = `
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Carregando...</span>
                    </div>
                    <div class="mt-2">Processando...</div>
                </div>
            `;
            document.body.appendChild(overlay);
        }
        overlay.style.display = 'flex';
    } else {
        if (overlay) {
            overlay.style.display = 'none';
        }
    }
}

function showButtonLoading(button) {
    if (button.dataset.loading === 'true') return;
    
    button.dataset.loading = 'true';
    button.dataset.originalText = button.innerHTML;
    button.disabled = true;
    button.innerHTML = '<i class="spinner-border spinner-border-sm me-2"></i>Processando...';
    
    // Fallback to reset button after 10 seconds
    setTimeout(() => {
        resetButtonLoading(button);
    }, 10000);
}

function resetButtonLoading(button) {
    if (button.dataset.loading !== 'true') return;
    
    button.dataset.loading = 'false';
    button.disabled = false;
    button.innerHTML = button.dataset.originalText || 'Enviar';
    delete button.dataset.originalText;
}

function confirmAction(message, callback) {
    if (confirm(message || 'Tem certeza que deseja continuar?')) {
        if (typeof callback === 'function') {
            callback();
        }
        return true;
    }
    return false;
}

function formatDate(dateString, format = 'dd/MM/yyyy') {
    if (!dateString) return '';
    
    const date = new Date(dateString);
    
    switch (format) {
        case 'dd/MM/yyyy':
            return date.toLocaleDateString('pt-BR');
        case 'dd/MM/yyyy HH:mm':
            return date.toLocaleString('pt-BR');
        default:
            return date.toLocaleDateString('pt-BR');
    }
}

function formatPhone(phone) {
    if (!phone) return '';
    
    const cleaned = phone.replace(/\D/g, '');
    
    if (cleaned.length === 11) {
        return cleaned.replace(/(\d{2})(\d{5})(\d{4})/, '($1) $2-$3');
    } else if (cleaned.length === 10) {
        return cleaned.replace(/(\d{2})(\d{4})(\d{4})/, '($1) $2-$3');
    }
    
    return phone;
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Export functions for global use
window.showAlert = showAlert;
window.showLoading = showLoading;
window.confirmAction = confirmAction;
window.formatDate = formatDate;
window.formatPhone = formatPhone;
window.debounce = debounce;
