
<?php

// Inicializar aplicação
require_once dirname(__DIR__) . '/config/config.php';

// Carregar variáveis de ambiente
loadEnv();

// Inicializar sessão
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Configuração de erro baseada no ambiente
if ($_ENV['APP_DEBUG'] ?? false) {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(E_ALL & ~E_NOTICE);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
}

// Autoloader simples
spl_autoload_register(function ($className) {
    $paths = [
        ROOT_PATH . '/src/models/',
        ROOT_PATH . '/src/controllers/',
        ROOT_PATH . '/src/services/',
        ROOT_PATH . '/src/utils/',
        ROOT_PATH . '/src/middleware/'
    ];
    
    foreach ($paths as $path) {
        $file = $path . $className . '.php';
        if (file_exists($file)) {
            require_once $file;
            return;
        }
    }
});

// Carregar dependências principais
require_once ROOT_PATH . '/config/database.php';
require_once ROOT_PATH . '/src/utils/Router.php';

// Criar instância do router
$router = new Router();

// Middleware de autenticação
$authController = new AuthController();

// === ROTAS PÚBLICAS ===

// Página inicial (redirecionar para login ou dashboard)
$router->get('/', function() {
    $authService = new AuthService();
    if ($authService->isLoggedIn()) {
        header('Location: /dashboard');
    } else {
        header('Location: /login');
    }
    exit;
});

// Rotas de autenticação
$router->get('/login', 'AuthController@showLogin');
$router->post('/login', 'AuthController@processLogin');
$router->get('/register', 'AuthController@showRegister');
$router->post('/register', 'AuthController@processRegister');
$router->get('/logout', 'AuthController@logout');

// === ROTAS PROTEGIDAS ===

// Dashboard
$router->middleware([$authController, 'requireAuth'])
    ->get('/dashboard', 'DashboardController@index');

$router->middleware([$authController, 'requireAuth'])
    ->get('/dashboard/reports', 'DashboardController@reports');

$router->middleware([$authController, 'requireAuth'])
    ->get('/dashboard/export', 'DashboardController@exportCSV');

// Perfil do usuário
$router->middleware([$authController, 'requireAuth'])
    ->get('/profile', 'AuthController@profile');

$router->middleware([$authController, 'requireAuth'])
    ->post('/profile', 'AuthController@updateProfile');

$router->middleware([$authController, 'requireAuth'])
    ->post('/profile/password', 'AuthController@changePassword');

// Currículos
$router->middleware([$authController, 'requireAuth'])
    ->get('/curriculos', 'CurriculumController@index');

$router->middleware([$authController, 'requireAuth'])
    ->get('/curriculos/create', 'CurriculumController@create');

$router->middleware([$authController, 'requireAuth'])
    ->post('/curriculos', 'CurriculumController@store');

$router->middleware([$authController, 'requireAuth'])
    ->get('/curriculos/{id}', 'CurriculumController@show');

$router->middleware([$authController, 'requireAuth'])
    ->get('/curriculos/{id}/edit', 'CurriculumController@edit');

$router->middleware([$authController, 'requireAuth'])
    ->post('/curriculos/{id}', 'CurriculumController@update');

$router->middleware([$authController, 'requireAuth'])
    ->post('/curriculos/{id}/delete', 'CurriculumController@delete');

// Grupos
$router->middleware([$authController, 'requireAuth'])
    ->get('/grupos', 'GroupController@index');

$router->middleware([$authController, 'requireAuth'])
    ->get('/grupos/create', 'GroupController@create');

$router->middleware([$authController, 'requireAuth'])
    ->post('/grupos', 'GroupController@store');

$router->middleware([$authController, 'requireAuth'])
    ->get('/grupos/{id}', 'GroupController@show');

$router->middleware([$authController, 'requireAuth'])
    ->get('/grupos/{id}/edit', 'GroupController@edit');

$router->middleware([$authController, 'requireAuth'])
    ->post('/grupos/{id}', 'GroupController@update');

$router->middleware([$authController, 'requireAuth'])
    ->post('/grupos/{id}/delete', 'GroupController@delete');

// Etiquetas
$router->middleware([$authController, 'requireAuth'])
    ->get('/etiquetas', 'TagController@index');

$router->middleware([$authController, 'requireAuth'])
    ->get('/etiquetas/create', 'TagController@create');

$router->middleware([$authController, 'requireAuth'])
    ->post('/etiquetas', 'TagController@store');

$router->middleware([$authController, 'requireAuth'])
    ->get('/etiquetas/{id}/edit', 'TagController@edit');

$router->middleware([$authController, 'requireAuth'])
    ->post('/etiquetas/{id}', 'TagController@update');

$router->middleware([$authController, 'requireAuth'])
    ->post('/etiquetas/{id}/delete', 'TagController@delete');

// Usuários (apenas admin e RH)
$router->middleware(function() use ($authController) {
    $authController->requirePermission('rh');
})->get('/usuarios', 'UserController@index');

$router->middleware(function() use ($authController) {
    $authController->requirePermission('admin');
})->get('/usuarios/create', 'UserController@create');

$router->middleware(function() use ($authController) {
    $authController->requirePermission('admin');
})->post('/usuarios', 'UserController@store');

$router->middleware(function() use ($authController) {
    $authController->requirePermission('rh');
})->get('/usuarios/{id}', 'UserController@show');

$router->middleware(function() use ($authController) {
    $authController->requirePermission('admin');
})->get('/usuarios/{id}/edit', 'UserController@edit');

$router->middleware(function() use ($authController) {
    $authController->requirePermission('admin');
})->post('/usuarios/{id}', 'UserController@update');

$router->middleware(function() use ($authController) {
    $authController->requirePermission('admin');
})->post('/usuarios/{id}/delete', 'UserController@delete');

// Upload de arquivos
$router->middleware([$authController, 'requireAuth'])
    ->post('/upload', 'FileController@upload');

$router->middleware([$authController, 'requireAuth'])
    ->get('/files/{file}', 'FileController@serve');

// API para busca de CEP
$router->middleware([$authController, 'requireAuth'])
    ->get('/api/cep/{cep}', 'ApiController@getCep');

// API para busca dinâmica
$router->middleware([$authController, 'requireAuth'])
    ->get('/api/search', 'ApiController@search');

// === TRATAMENTO DE ERROS ===

// Definir handler para rota não encontrada
$router->setNotFoundHandler(function() {
    http_response_code(404);
    $title = 'Página não encontrada';
    require VIEWS_PATH . '/errors/404.php';
});

// Handler de exceções global
set_exception_handler(function($exception) {
    logMessage("Exceção não tratada: " . $exception->getMessage(), 'ERROR');
    
    if ($_ENV['APP_DEBUG'] ?? false) {
        echo "<pre>";
        echo "Erro: " . $exception->getMessage() . "\n";
        echo "Arquivo: " . $exception->getFile() . "\n";
        echo "Linha: " . $exception->getLine() . "\n";
        echo "Stack trace:\n" . $exception->getTraceAsString();
        echo "</pre>";
    } else {
        http_response_code(500);
        $title = 'Erro interno';
        require VIEWS_PATH . '/errors/500.php';
    }
});

// Handler de erros global
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    if (!(error_reporting() & $errno)) {
        return;
    }
    
    $errorMessage = "Erro PHP [$errno]: $errstr em $errfile na linha $errline";
    logMessage($errorMessage, 'ERROR');
    
    if ($_ENV['APP_DEBUG'] ?? false) {
        echo "<div style='background: #f8d7da; color: #721c24; padding: 10px; margin: 10px; border: 1px solid #f5c6cb; border-radius: 5px;'>";
        echo "<strong>PHP Error:</strong> $errstr<br>";
        echo "<strong>File:</strong> $errfile<br>";
        echo "<strong>Line:</strong> $errline";
        echo "</div>";
    }
    
    return true;
});

// === EXECUTAR ROUTER ===
try {
    $router->run();
} catch (Exception $e) {
    logMessage("Erro no roteamento: " . $e->getMessage(), 'ERROR');
    
    if ($_ENV['APP_DEBUG'] ?? false) {
        throw $e;
    } else {
        http_response_code(500);
        $title = 'Erro interno';
        require VIEWS_PATH . '/errors/500.php';
    }
}
