
<?php

// Definições de configuração da aplicação
define('APP_NAME', 'Gerenciador de Currículos');
define('APP_VERSION', '1.0.0');
define('APP_URL', $_ENV['APP_URL'] ?? 'http://localhost:8080');

// Configurações de segurança
define('SESSION_TIMEOUT', 3600); // 1 hora
define('PASSWORD_MIN_LENGTH', 6);
define('UPLOAD_MAX_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_FILE_TYPES', ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png']);

// Diretórios
define('ROOT_PATH', dirname(__DIR__));
define('PUBLIC_PATH', ROOT_PATH . '/public');
define('UPLOAD_PATH', PUBLIC_PATH . '/uploads');
define('VIEWS_PATH', ROOT_PATH . '/src/views');
define('MODELS_PATH', ROOT_PATH . '/src/models');
define('CONTROLLERS_PATH', ROOT_PATH . '/src/controllers');

// Configurações de paginação
define('ITEMS_PER_PAGE', 20);
define('MAX_PAGINATION_LINKS', 10);

// Configurações de email (futuro)
define('SMTP_HOST', $_ENV['SMTP_HOST'] ?? '');
define('SMTP_PORT', $_ENV['SMTP_PORT'] ?? 587);
define('SMTP_USER', $_ENV['SMTP_USER'] ?? '');
define('SMTP_PASS', $_ENV['SMTP_PASS'] ?? '');

// Configurações de log
define('LOG_LEVEL', $_ENV['LOG_LEVEL'] ?? 'ERROR');
define('LOG_FILE', ROOT_PATH . '/logs/app.log');

// Tipos de usuário
define('USER_TYPES', [
    'admin' => 'Administrador',
    'rh' => 'RH',
    'recrutador' => 'Recrutador'
]);

// Status de currículos
define('CURRICULUM_STATUS', [
    'novo' => 'Novo',
    'triagem' => 'Em Triagem',
    'aprovado' => 'Aprovado',
    'rejeitado' => 'Rejeitado',
    'entrevista' => 'Em Entrevista',
    'contratado' => 'Contratado'
]);

// Função para carregar variáveis de ambiente
function loadEnv($path = null) {
    $envFile = $path ?? ROOT_PATH . '/.env';
    
    if (!file_exists($envFile)) {
        return;
    }
    
    $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    
    foreach ($lines as $line) {
        if (strpos(trim($line), '#') === 0) {
            continue; // Comentário
        }
        
        list($name, $value) = explode('=', $line, 2);
        $name = trim($name);
        $value = trim($value);
        
        if (!array_key_exists($name, $_ENV)) {
            $_ENV[$name] = $value;
            $_SERVER[$name] = $value;
            putenv("$name=$value");
        }
    }
}

// Função para debug
function debug($data, $die = false) {
    if ($_ENV['APP_DEBUG'] ?? false) {
        echo '<pre>';
        print_r($data);
        echo '</pre>';
        
        if ($die) {
            die();
        }
    }
}

// Função para log
function logMessage($message, $level = 'INFO') {
    $logDir = dirname(LOG_FILE);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message" . PHP_EOL;
    
    file_put_contents(LOG_FILE, $logEntry, FILE_APPEND | LOCK_EX);
}

// Função para sanitizar dados
function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

// Função para validar email
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Função para formatar telefone
function formatPhone($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    if (strlen($phone) == 11) {
        return '(' . substr($phone, 0, 2) . ') ' . substr($phone, 2, 5) . '-' . substr($phone, 7);
    } elseif (strlen($phone) == 10) {
        return '(' . substr($phone, 0, 2) . ') ' . substr($phone, 2, 4) . '-' . substr($phone, 6);
    }
    
    return $phone;
}

// Função para formatar data brasileira
function formatDate($date, $format = 'd/m/Y') {
    if (empty($date) || $date === '0000-00-00') {
        return '';
    }
    
    return date($format, strtotime($date));
}

// Função para gerar token CSRF
function generateCsrfToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Função para verificar token CSRF
function verifyCsrfToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}
